<?php

namespace Database\Seeders;

use App\Models\DetailPresensiMadrasah;
use App\Models\JenisPresensiMadrasah;
use App\Models\Kelas;
use App\Models\PresensiMadrasah;
use App\Models\TahunAjaran;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Database\Seeder;

class PresensiMadrasahSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get active tahun ajaran
        $tahunAjaran = TahunAjaran::where('is_aktif', true)->first();
        if (! $tahunAjaran) {
            if ($this->command) {
                $this->command->error('No active TahunAjaran found. Cannot seed PresensiMadrasah.');
            }

            return;
        }

        // Get all jenis presensi
        $jenisPresensiList = JenisPresensiMadrasah::where('is_active', true)->get();
        if ($jenisPresensiList->isEmpty()) {
            if ($this->command) {
                $this->command->error('No JenisPresensiMadrasah found. Cannot seed PresensiMadrasah.');
            }

            return;
        }

        // Get all kelas
        $kelasList = Kelas::all();
        if ($kelasList->isEmpty()) {
            if ($this->command) {
                $this->command->error('No Kelas found. Cannot seed PresensiMadrasah.');
            }

            return;
        }

        // Get a user for petugas
        $user = User::first();
        if (! $user) {
            if ($this->command) {
                $this->command->error('No User found. Cannot seed PresensiMadrasah.');
            }

            return;
        }

        $totalPresensi = 0;
        $totalDetail = 0;

        // For each kelas, create 6 presensi
        foreach ($kelasList as $kelas) {
            if ($this->command) {
                $this->command->info("Creating presensi for class: {$kelas->nama} (Madrasah: {$kelas->madrasah->nama})");
            }

            // Generate 6 random dates (mix of 1 week and 1 month ago)
            $dates = $this->generateRandomDates(6);

            foreach ($dates as $tanggal) {
                // Random jenis presensi
                $jenisPresensi = $jenisPresensiList->random();

                // Create PresensiMadrasah
                $presensi = PresensiMadrasah::create([
                    'tanggal' => $tanggal,
                    'tahun_ajaran_id' => $tahunAjaran->id,
                    'jenis_presensi_madrasah_id' => $jenisPresensi->id,
                    'kelas_id' => $kelas->id,
                    'user_id' => $user->id,
                    'keterangan' => fake()->optional(0.3)->sentence(),
                    'is_locked' => fake()->boolean(50), // 50% locked
                ]);

                $totalPresensi++;

                // Get all siswa in this kelas
                $siswaList = $kelas->siswas;

                // Create DetailPresensiMadrasah for each siswa
                foreach ($siswaList as $siswa) {
                    // Random status with realistic distribution
                    $status = $this->getRandomStatus();

                    DetailPresensiMadrasah::create([
                        'presensi_madrasah_id' => $presensi->id,
                        'siswa_id' => $siswa->id,
                        'status' => $status,
                        'keterangan' => $status !== 'hadir' ? fake()->optional(0.5)->sentence() : null,
                    ]);

                    $totalDetail++;
                }
            }
        }

        if ($this->command) {
            $this->command->info("Successfully created {$totalPresensi} presensi with {$totalDetail} detail records.");
        }
    }

    /**
     * Generate random dates (mix of 1 week and 1 month ago)
     */
    private function generateRandomDates(int $count): array
    {
        $dates = [];
        $now = Carbon::now();

        for ($i = 0; $i < $count; $i++) {
            // 50% chance for 1 week ago, 50% for 1 month ago
            if (fake()->boolean(50)) {
                // 1 week ago (1-7 days)
                $daysAgo = fake()->numberBetween(1, 7);
            } else {
                // 1 month ago (8-30 days)
                $daysAgo = fake()->numberBetween(8, 30);
            }

            $date = $now->copy()->subDays($daysAgo);

            // Skip weekends (Saturday = 6, Sunday = 0)
            while ($date->dayOfWeek === 0 || $date->dayOfWeek === 6) {
                $date->subDay();
            }

            $dates[] = $date->format('Y-m-d');
        }

        // Remove duplicates and sort
        $dates = array_unique($dates);
        sort($dates);

        // If we have less than required due to duplicates, add more
        while (count($dates) < $count) {
            $daysAgo = fake()->numberBetween(1, 30);
            $date = $now->copy()->subDays($daysAgo);

            while ($date->dayOfWeek === 0 || $date->dayOfWeek === 6) {
                $date->subDay();
            }

            $dateStr = $date->format('Y-m-d');
            if (! in_array($dateStr, $dates)) {
                $dates[] = $dateStr;
            }
        }

        sort($dates);

        return array_slice($dates, 0, $count);
    }

    /**
     * Get random status with realistic distribution
     * 85% hadir, 8% sakit, 5% izin, 2% alpha
     */
    private function getRandomStatus(): string
    {
        $rand = fake()->numberBetween(1, 100);

        if ($rand <= 85) {
            return 'hadir';
        } elseif ($rand <= 93) {
            return 'sakit';
        } elseif ($rand <= 98) {
            return 'izin';
        } else {
            return 'alpha';
        }
    }
}
