<?php

namespace App\Traits;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;

trait HasMadrasahScope
{
    /**
     * Flag to prevent recursive scope application
     */
    private static bool $scopeApplying = false;

    /**
     * Apply madrasah scoping to the query based on authenticated user's role
     */
    public static function bootHasMadrasahScope(): void
    {
        static::addGlobalScope('madrasah', function (Builder $builder) {
            // Prevent recursive calls that cause infinite loops
            if (self::$scopeApplying) {
                return;
            }

            self::$scopeApplying = true;

            try {
                $user = Auth::user();

                if (! $user) {
                    return;
                }

                // Super admin, pimpinan, and operator can see all madrasahs
                if ($user->hasRole(['super_admin', 'pimpinan', 'operator'])) {
                    return;
                }

                // Get user's madrasah ID
                $madrasahId = $user->getMadrasahId();

                if (! $madrasahId) {
                    // If user has no madrasah, show nothing
                    $builder->whereRaw('1 = 0');

                    return;
                }

                // Apply madrasah filter
                // Include records with matching madrasah_id OR NULL (global records)
                $builder->where(function ($query) use ($madrasahId) {
                    $query->where('madrasah_id', $madrasahId)
                        ->orWhereNull('madrasah_id');
                });
            } finally {
                // Always reset the flag, even if an exception occurs
                self::$scopeApplying = false;
            }
        });
    }
}
