<?php

namespace App\Services;

use App\Models\DetailPresensiMadrasah;
use App\Models\JenisPresensiMadrasah;
use App\Models\Kelas;
use App\Models\PresensiMadrasah;
use App\Models\TahunAjaran;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PresensiMadrasahService
{
    /**
     * Generate attendance for a specific class and attendance type.
     */
    public function generateForKelas(
        Kelas $kelas,
        JenisPresensiMadrasah $jenisPresensi,
        Carbon $tanggal,
        User $user
    ): array {
        try {
            DB::beginTransaction();

            $tahunAjaran = TahunAjaran::where('is_aktif', true)->first();

            if (! $tahunAjaran) {
                return [
                    'success' => false,
                    'message' => 'Tahun ajaran aktif tidak ditemukan',
                ];
            }

            // Check if attendance already exists
            $existingPresensi = PresensiMadrasah::where([
                'tanggal' => $tanggal->toDateString(),
                'kelas_id' => $kelas->id,
                'jenis_presensi_madrasah_id' => $jenisPresensi->id,
                'tahun_ajaran_id' => $tahunAjaran->id,
            ])->first();

            if ($existingPresensi) {
                return [
                    'success' => false,
                    'message' => "Presensi {$jenisPresensi->nama} untuk kelas {$kelas->nama} sudah ada",
                    'presensi' => $existingPresensi,
                ];
            }

            // Create attendance header
            $presensi = PresensiMadrasah::create([
                'tanggal' => $tanggal->toDateString(),
                'tahun_ajaran_id' => $tahunAjaran->id,
                'jenis_presensi_madrasah_id' => $jenisPresensi->id,
                'kelas_id' => $kelas->id,
                'user_id' => $user->id,
                'is_locked' => false,
                'keterangan' => 'Auto-generated by Quick Generate',
            ]);

            // Generate detail for all students in the class
            $siswas = $kelas->siswas;
            $detailCount = 0;

            foreach ($siswas as $siswa) {
                DetailPresensiMadrasah::create([
                    'presensi_madrasah_id' => $presensi->id,
                    'siswa_id' => $siswa->id,
                    'status' => 'hadir', // Default status
                    'keterangan' => null,
                ]);
                $detailCount++;
            }

            DB::commit();

            return [
                'success' => true,
                'message' => "Berhasil generate presensi {$jenisPresensi->nama} untuk {$detailCount} siswa",
                'presensi' => $presensi,
                'siswa_count' => $detailCount,
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error generating attendance for class', [
                'kelas_id' => $kelas->id,
                'jenis_presensi_id' => $jenisPresensi->id,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => 'Terjadi kesalahan: '.$e->getMessage(),
            ];
        }
    }

    /**
     * Generate attendance for all classes in a madrasah.
     */
    public function generateAllForMadrasah(int $madrasahId, Carbon $tanggal, int $userId): array
    {
        try {
            $user = User::find($userId);
            if (! $user) {
                return [
                    'success' => false,
                    'message' => 'User tidak ditemukan',
                ];
            }

            // Get all active classes for the madrasah
            $kelas = Kelas::where('madrasah_id', $madrasahId)
                ->with('siswas')
                ->get();

            if ($kelas->isEmpty()) {
                return [
                    'success' => false,
                    'message' => 'Tidak ada kelas ditemukan untuk madrasah ini',
                ];
            }

            // Get all active attendance types for the madrasah
            // Include types that are specific to this madrasah OR types that apply to all madrasahs (madrasah_id = null)
            $jenisPresensi = JenisPresensiMadrasah::where('is_active', true)
                ->where(function ($query) use ($madrasahId) {
                    $query->where('madrasah_id', $madrasahId)
                        ->orWhereNull('madrasah_id');
                })
                ->get();

            if ($jenisPresensi->isEmpty()) {
                return [
                    'success' => false,
                    'message' => 'Tidak ada jenis presensi aktif ditemukan untuk madrasah ini',
                ];
            }

            $results = [];
            $totalSuccess = 0;
            $totalSkipped = 0;
            $totalErrors = 0;

            foreach ($kelas as $kelasItem) {
                foreach ($jenisPresensi as $jenis) {
                    $result = $this->generateForKelas($kelasItem, $jenis, $tanggal, $user);

                    if ($result['success']) {
                        $totalSuccess++;
                        $results[] = [
                            'kelas' => $kelasItem->nama,
                            'jenis' => $jenis->nama,
                            'status' => 'success',
                            'siswa_count' => $result['siswa_count'],
                        ];
                    } else {
                        if (str_contains($result['message'], 'sudah ada')) {
                            $totalSkipped++;
                            $results[] = [
                                'kelas' => $kelasItem->nama,
                                'jenis' => $jenis->nama,
                                'status' => 'skipped',
                                'message' => $result['message'],
                            ];
                        } else {
                            $totalErrors++;
                            $results[] = [
                                'kelas' => $kelasItem->nama,
                                'jenis' => $jenis->nama,
                                'status' => 'error',
                                'message' => $result['message'],
                            ];
                        }
                    }
                }
            }

            $summary = "Hasil generate: {$totalSuccess} berhasil, {$totalSkipped} dilewati, {$totalErrors} error";

            return [
                'success' => true,
                'message' => $summary,
                'results' => $results,
                'summary' => [
                    'success' => $totalSuccess,
                    'skipped' => $totalSkipped,
                    'errors' => $totalErrors,
                    'total_kelas' => $kelas->count(),
                    'total_jenis' => $jenisPresensi->count(),
                ],
            ];
        } catch (\Exception $e) {
            Log::error('Error generating attendance for madrasah', [
                'madrasah_id' => $madrasahId,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => 'Terjadi kesalahan: '.$e->getMessage(),
            ];
        }
    }

    /**
     * Get attendance summary for a madrasah on a specific date.
     */
    public function getAttendanceSummary(int $madrasahId, Carbon $tanggal): Collection
    {
        $kelas = Kelas::where('madrasah_id', $madrasahId)
            ->with(['siswas'])
            ->get();

        // Get all active attendance types that apply to this madrasah
        // Include types specific to this madrasah OR types that apply to all madrasahs (madrasah_id = null)
        $jenisPresensi = JenisPresensiMadrasah::where('is_active', true)
            ->where(function ($query) use ($madrasahId) {
                $query->where('madrasah_id', $madrasahId)
                    ->orWhereNull('madrasah_id');
            })
            ->get();

        $summary = collect();

        foreach ($kelas as $kelasItem) {
            foreach ($jenisPresensi as $jenis) {
                $presensi = PresensiMadrasah::where([
                    'tanggal' => $tanggal->toDateString(),
                    'kelas_id' => $kelasItem->id,
                    'jenis_presensi_madrasah_id' => $jenis->id,
                ])->first();

                $summary->push([
                    'kelas' => $kelasItem->nama,
                    'jenis' => $jenis->nama,
                    'siswa_count' => $kelasItem->siswas->count(),
                    'has_attendance' => (bool) $presensi,
                    'is_locked' => $presensi?->is_locked ?? false,
                    'presensi_id' => $presensi?->id,
                    'detail_count' => $presensi?->detailPresensiMadrasahs()?->count() ?? 0,
                ]);
            }
        }

        return $summary;
    }

    /**
     * Generate attendance for selected classes in a madrasah.
     */
    public function generateForSelectedKelas(int $madrasahId, Carbon $tanggal, int $userId, array $kelasIds, array $jenisPresensiIds = []): array
    {
        try {
            $user = User::find($userId);
            if (! $user) {
                return [
                    'success' => false,
                    'message' => 'User tidak ditemukan',
                ];
            }

            if (empty($kelasIds)) {
                return [
                    'success' => false,
                    'message' => 'Pilih minimal 1 kelas',
                ];
            }

            // Get selected classes for the madrasah
            $kelas = Kelas::where('madrasah_id', $madrasahId)
                ->whereIn('id', $kelasIds)
                ->with('siswas')
                ->get();

            if ($kelas->isEmpty()) {
                return [
                    'success' => false,
                    'message' => 'Kelas yang dipilih tidak ditemukan',
                ];
            }

            // Get active tahun ajaran
            $tahunAjaran = TahunAjaran::where('is_aktif', true)->first();
            if (! $tahunAjaran) {
                return [
                    'success' => false,
                    'message' => 'Tidak ada tahun ajaran yang aktif',
                ];
            }

            // Get selected jenis presensi or all active ones
            $query = JenisPresensiMadrasah::where('is_active', true)
                ->where(function ($query) use ($madrasahId) {
                    $query->where('madrasah_id', $madrasahId)
                        ->orWhereNull('madrasah_id');
                });

            if (! empty($jenisPresensiIds)) {
                $query->whereIn('id', $jenisPresensiIds);
            }

            $jenisPresensi = $query->get();

            if ($jenisPresensi->isEmpty()) {
                return [
                    'success' => false,
                    'message' => 'Tidak ada jenis presensi aktif ditemukan untuk madrasah ini',
                ];
            }

            $results = [];
            $totalSuccess = 0;
            $totalSkipped = 0;
            $totalErrors = 0;

            foreach ($kelas as $kelasItem) {
                foreach ($jenisPresensi as $jenis) {
                    $result = $this->generateForKelas($kelasItem, $jenis, $tanggal, $user);

                    if ($result['success']) {
                        $totalSuccess++;
                        $results[] = [
                            'kelas' => $kelasItem->nama,
                            'jenis' => $jenis->nama,
                            'status' => 'success',
                            'siswa_count' => $result['siswa_count'],
                        ];
                    } else {
                        if (str_contains($result['message'], 'sudah ada')) {
                            $totalSkipped++;
                            $results[] = [
                                'kelas' => $kelasItem->nama,
                                'jenis' => $jenis->nama,
                                'status' => 'skipped',
                                'message' => $result['message'],
                            ];
                        } else {
                            $totalErrors++;
                            $results[] = [
                                'kelas' => $kelasItem->nama,
                                'jenis' => $jenis->nama,
                                'status' => 'error',
                                'message' => $result['message'],
                            ];
                        }
                    }
                }
            }

            $summary = "Hasil generate: {$totalSuccess} berhasil, {$totalSkipped} dilewati, {$totalErrors} error";

            return [
                'success' => true,
                'message' => $summary,
                'results' => $results,
                'summary' => [
                    'success' => $totalSuccess,
                    'skipped' => $totalSkipped,
                    'errors' => $totalErrors,
                    'total_kelas' => $kelas->count(),
                    'total_jenis' => $jenisPresensi->count(),
                ],
            ];
        } catch (\Exception $e) {
            Log::error('Error generating attendance for selected classes', [
                'madrasah_id' => $madrasahId,
                'kelas_ids' => $kelasIds,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => 'Terjadi kesalahan: '.$e->getMessage(),
            ];
        }
    }
}
