<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;

use Filament\Models\Contracts\FilamentUser;
use Filament\Models\Contracts\HasAvatar;
use Filament\Panel;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\Storage;
use Jeffgreco13\FilamentBreezy\Traits\TwoFactorAuthenticatable;
use Laravel\Sanctum\HasApiTokens;
use Spatie\Permission\Traits\HasRoles;

class User extends Authenticatable implements FilamentUser, HasAvatar, MustVerifyEmail
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasApiTokens, HasFactory, HasRoles, Notifiable, TwoFactorAuthenticatable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'avatar_url',
        'madrasah_id',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
        ];
    }

    // public function getFilamentAvatarUrl(): ?string
    // {
    //     return asset($this->avatar_url);
    // }

    public function getFilamentAvatarUrl(): ?string
    {
        return $this->avatar_url ? Storage::url($this->avatar_url) : null;
    }

    /**
     * Get the kelas where this user is the wali kelas (homeroom teacher)
     */
    public function kelasAsWali(): HasOne
    {
        return $this->hasOne(Kelas::class, 'wali_kelas_id');
    }

    /**
     * Accessor for kelasAsWali property to prevent undefined property errors
     * This is especially important when using impersonate or Filament Shield
     */
    public function getKelasAsWaliAttribute()
    {
        // Check if relationship is already loaded
        if ($this->relationLoaded('kelasAsWali')) {
            return $this->getRelation('kelasAsWali');
        }

        // Load and return the relationship
        return $this->kelasAsWali()->first();
    }

    /**
     * Get the madrasah that this user belongs to
     */
    public function madrasah(): BelongsTo
    {
        return $this->belongsTo(Madrasah::class);
    }

    /**
     * Get the user's madrasah ID (direct or through kelas/kamar)
     */
    public function getMadrasahId(): ?int
    {
        // Direct madrasah assignment (for kepala sekolah, etc.)
        if ($this->madrasah_id) {
            return $this->madrasah_id;
        }

        // Through kelas (for walikelas) - WITHOUT global scope to prevent circular dependency
        if ($kelas = Kelas::withoutGlobalScope('madrasah')
            ->where('wali_kelas_id', $this->id)
            ->first()) {
            return $kelas->madrasah_id;
        }

        // Through kamar (for wali kamar) - WITHOUT global scope to prevent circular dependency
        if ($kamar = Kamar::withoutGlobalScope('madrasah')
            ->where('wali_kamar_id', $this->id)
            ->first()) {
            return $kamar->madrasah_id;
        }

        return null;
    }

    /**
     * Get the siswa linked to this user (for Ortu role)
     */
    public function siswa(): HasOne
    {
        return $this->hasOne(Siswa::class);
    }

    public function canAccessPanel(Panel $panel): bool
    {
        return true;
    }
}
