<?php

namespace App\Models;

use App\Traits\HasMadrasahScope;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class Siswa extends Model
{
    use HasFactory, SoftDeletes, HasMadrasahScope;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'madrasah_id',
        'kelas_id',
        'nisn',
        'nis',
        'nik',
        'nama',
        'jenis_kelamin',
        'tempat_lahir',
        'tanggal_lahir',
        'status_tempat_tinggal',
        'kamar_id',
        'user_id',
        'no_kk',
        'nama_ayah',
        'nik_ayah',
        'pendidikan_ayah',
        'pekerjaan_ayah',
        'nama_ibu',
        'nik_ibu',
        'pendidikan_ibu',
        'pekerjaan_ibu',
        'no_hp_ortu',
        'alamat',
        'provinsi',
        'kabupaten_kota',
        'kecamatan',
        'desa_kelurahan',
        'kode_pos',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'id' => 'integer',
        'madrasah_id' => 'integer',
        'kelas_id' => 'integer',
        'tanggal_lahir' => 'date',
        'kamar_id' => 'integer',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function madrasah(): BelongsTo
    {
        return $this->belongsTo(Madrasah::class);
    }

    public function kelas(): BelongsTo
    {
        return $this->belongsTo(Kelas::class);
    }

    public function kamar(): BelongsTo
    {
        return $this->belongsTo(Kamar::class);
    }

    public function detailPresensiMadrasahs(): HasMany
    {
        return $this->hasMany(DetailPresensiMadrasah::class);
    }

    public function detailPresensiAsramas(): HasMany
    {
        return $this->hasMany(DetailPresensiAsrama::class);
    }

    public function riwayatKelas(): HasMany
    {
        return $this->hasMany(RiwayatKelas::class);
    }

    /**
     * Scope to filter only active students
     * Active = has active RiwayatKelas OR has no RiwayatKelas at all (backward compatibility)
     */
    public function scopeAktif($query)
    {
        return $query->where(function ($q) {
            // Has active RiwayatKelas
            $q->whereHas('riwayatKelas', function ($subQ) {
                $subQ->where('status', 'aktif');
            })
            // OR has no RiwayatKelas at all (old students)
            ->orWhereDoesntHave('riwayatKelas');
        });
    }

    /**
     * Scope to filter alumni students (has RiwayatKelas but no active status)
     */
    public function scopeAlumni($query)
    {
        return $query->whereHas('riwayatKelas') // Must have RiwayatKelas
            ->whereDoesntHave('riwayatKelas', function ($q) {
                $q->where('status', 'aktif');
            });
    }

    /**
     * Check if student is active
     * Active = has active RiwayatKelas OR has no RiwayatKelas at all
     */
    public function isAktif(): bool
    {
        $hasRiwayat = $this->riwayatKelas()->exists();
        
        if (!$hasRiwayat) {
            // No RiwayatKelas = old student, considered active
            return true;
        }
        
        // Has RiwayatKelas, check if any is active
        return $this->riwayatKelas()->where('status', 'aktif')->exists();
    }

    /**
     * Check if student is alumni
     * Alumni = has RiwayatKelas but no active status
     */
    public function isAlumni(): bool
    {
        $hasRiwayat = $this->riwayatKelas()->exists();
        
        if (!$hasRiwayat) {
            // No RiwayatKelas = not alumni
            return false;
        }
        
        // Has RiwayatKelas but no active status = alumni
        return !$this->riwayatKelas()->where('status', 'aktif')->exists();
    }
}
