<?php

namespace App\Filament\Widgets;

use App\Services\PresensiMadrasahService;
use BezhanSalleh\FilamentShield\Traits\HasWidgetShield;
use Carbon\Carbon;
use Filament\Actions\Action;
use Filament\Actions\Concerns\InteractsWithActions;
use Filament\Actions\Contracts\HasActions;
use Filament\Forms;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Notifications\Notification;
use Filament\Widgets\Widget;
use Illuminate\Support\Facades\Auth;

class QuickGeneratePresensiWidget extends Widget implements HasActions, HasForms
{
    use HasWidgetShield, InteractsWithActions, InteractsWithForms;

    protected static string $view = 'filament.widgets.quick-generate-presensi-widget';

    protected static ?int $sort = 1;

    protected static ?string $heading = 'Quick Generate Presensi Hari Ini';

    protected int|string|array $columnSpan = 'full';

    public function mount(): void
    {
        // Check if user has access
        $user = Auth::user();
        if (! $user || ! $user->hasAnyRole(['wakil kepala', 'Wakil Kepala', 'kepala sekolah', 'operator'])) {
            abort(403, 'Tidak memiliki akses ke widget ini');
        }
    }

    public function generateAllAttendanceAction(): Action
    {
        return Action::make('generateAllAttendance')
            ->label('Generate Semua Absen Hari Ini')
            ->icon('heroicon-o-calendar-days')
            ->color('primary')
            ->size('lg')
            ->form([
                Forms\Components\MultiSelect::make('kelas_ids')
                    ->label('Pilih Kelas')
                    ->options(function () {
                        $user = Auth::user();
                        $madrasahId = $user->getMadrasahId();

                        return \App\Models\Kelas::where('madrasah_id', $madrasahId)
                            ->orderBy('nama')
                            ->pluck('nama', 'id')
                            ->toArray();
                    })
                    ->searchable()
                    ->preload()
                    ->required()
                    ->default(function () {
                        // Default: semua kelas di madrasah user
                        $user = Auth::user();
                        $madrasahId = $user->getMadrasahId();

                        return \App\Models\Kelas::where('madrasah_id', $madrasahId)
                            ->pluck('id')
                            ->toArray();
                    })
                    ->helperText('Default: semua kelas. Uncheck untuk exclude kelas tertentu.'),
                Forms\Components\Select::make('jenis_presensi_id')
                    ->label('Pilih Jenis Presensi')
                    ->options(function () {
                        $user = Auth::user();
                        $madrasahId = $user->getMadrasahId();

                        return \App\Models\JenisPresensiMadrasah::where('is_active', true)
                            ->where(function ($query) use ($madrasahId) {
                                $query->where('madrasah_id', $madrasahId)
                                    ->orWhereNull('madrasah_id');
                            })
                            ->orderBy('nama')
                            ->pluck('nama', 'id')
                            ->toArray();
                    })
                    ->searchable()
                    ->preload()
                    ->required()
                    ->default(function () {
                        // Default: first active jenis presensi
                        $user = Auth::user();
                        $madrasahId = $user->getMadrasahId();

                        return \App\Models\JenisPresensiMadrasah::where('is_active', true)
                            ->where(function ($query) use ($madrasahId) {
                                $query->where('madrasah_id', $madrasahId)
                                    ->orWhereNull('madrasah_id');
                            })
                            ->orderBy('nama')
                            ->first()
                            ?->id;
                    })
                    ->helperText('Pilih satu jenis presensi untuk di-generate'),
            ])
            ->modalHeading('Generate Presensi Hari Ini')
            ->modalDescription('Pilih kelas yang ingin di-generate absennya. Presensi yang sudah ada akan dilewati.')
            ->modalSubmitActionLabel('Generate')
            ->action(function (array $data) {
                $user = Auth::user();
                $madrasahId = $user->getMadrasahId();

                if (! $madrasahId) {
                    Notification::make()
                        ->title('Error')
                        ->body('Madrasah ID tidak ditemukan untuk user ini')
                        ->danger()
                        ->send();

                    return;
                }

                $kelasIds = $data['kelas_ids'] ?? [];
                $jenisPresensiId = $data['jenis_presensi_id'] ?? null;

                if (empty($kelasIds)) {
                    Notification::make()
                        ->title('Validasi')
                        ->body('Pilih minimal 1 kelas')
                        ->warning()
                        ->send();

                    return;
                }

                if (! $jenisPresensiId) {
                    Notification::make()
                        ->title('Validasi')
                        ->body('Pilih 1 jenis presensi')
                        ->warning()
                        ->send();

                    return;
                }

                $service = new PresensiMadrasahService;
                $result = $service->generateForSelectedKelas($madrasahId, Carbon::today(), $user->id, $kelasIds, [$jenisPresensiId]);

                if ($result['success']) {
                    $summary = $result['summary'];
                    $message = "Generate selesai! Berhasil: {$summary['success']}, Dilewati: {$summary['skipped']}, Error: {$summary['errors']}";

                    Notification::make()
                        ->title('Generate Berhasil!')
                        ->body($message)
                        ->success()
                        ->duration(8000)
                        ->send();

                    // Detailed notification if there are errors
                    if ($summary['errors'] > 0) {
                        $errorDetails = collect($result['results'])
                            ->filter(fn ($r) => $r['status'] === 'error')
                            ->pluck('message')
                            ->unique()
                            ->join(', ');

                        Notification::make()
                            ->title('Ada Error')
                            ->body("Detail error: {$errorDetails}")
                            ->warning()
                            ->duration(10000)
                            ->send();
                    }
                } else {
                    Notification::make()
                        ->title('Generate Gagal')
                        ->body($result['message'])
                        ->danger()
                        ->send();
                }

                // Refresh the widget to show updated data
                $this->dispatch('refreshWidget');
            });
    }

    protected function getAttendanceSummary(): array
    {
        $user = Auth::user();
        $madrasahId = $user->getMadrasahId();

        if (! $madrasahId) {
            return [
                'error' => 'Madrasah ID tidak ditemukan',
                'summary' => collect(),
                'date' => Carbon::today()->format('d F Y'),
            ];
        }

        // Get active tahun ajaran
        $tahunAjaran = \App\Models\TahunAjaran::where('is_aktif', true)->first();
        if (! $tahunAjaran) {
            return [
                'error' => 'Tidak ada tahun ajaran yang aktif',
                'summary' => collect(),
                'date' => Carbon::today()->format('d F Y'),
            ];
        }

        $service = new PresensiMadrasahService;
        $summary = $service->getAttendanceSummary($madrasahId, Carbon::today());

        return [
            'summary' => $summary,
            'date' => Carbon::today()->format('d F Y'),
            'tahun_ajaran' => $tahunAjaran->tahun,
        ];
    }

    public function refreshAction(): Action
    {
        return Action::make('refresh')
            ->label('Refresh')
            ->icon('heroicon-o-arrow-path')
            ->color('gray')
            ->action(function () {
                $this->dispatch('refreshWidget');

                Notification::make()
                    ->title('Data direfresh')
                    ->success()
                    ->send();
            });
    }

    protected function getViewData(): array
    {
        return $this->getAttendanceSummary();
    }

    public static function canView(): bool
    {
        $user = Auth::user();

        return $user && $user->hasAnyRole(['wakil kepala', 'Wakil Kepala', 'kepala sekolah', 'operator']);
    }
}
