<?php

namespace App\Filament\Widgets;

use App\Models\JenisPresensiMadrasah;
use App\Models\Kelas;
use App\Models\PresensiMadrasah;
use App\Models\TahunAjaran;
use BezhanSalleh\FilamentShield\Traits\HasWidgetShield;
use Carbon\Carbon;
use Filament\Widgets\Concerns\InteractsWithPageFilters;
use Filament\Widgets\StatsOverviewWidget as BaseWidget;
use Filament\Widgets\StatsOverviewWidget\Stat;
use Illuminate\Support\Facades\Auth;

class PresensiMadrasahWidget extends BaseWidget
{
    use \App\Traits\HasStrictWidgetShield, InteractsWithPageFilters;

    protected static ?int $sort = 9;

    protected function getStats(): array
    {
        $user = Auth::user();
        $kelasId = null;

        // If user is a walikelas, automatically find their class
        if ($user->hasRole('walikelas')) {
            // Use global scope - it will automatically filter by madrasah
            $kelas = Kelas::where('wali_kelas_id', $user->id)->first();
            if ($kelas) {
                $kelasId = $kelas->id;
            }
        } else {
            // For other roles, use the page filter or the first class as a default
            // Global scope will automatically filter by madrasah
            $kelasId = $this->filters['kelas_id'] ?? Kelas::first()?->id;
        }

        if (! $kelasId) {
            // If user is walikelas but no class is assigned, or no classes exist for other roles
            $message = $user->hasRole('walikelas') ? 'Anda tidak ditugaskan di kelas manapun.' : 'Belum ada data kelas.';

            return [
                Stat::make('Info', $message)
                    ->color('gray'),
            ];
        }

        // The rest of the original logic
        $today = Carbon::today();
        $tahunAjaran = TahunAjaran::where('is_aktif', true)->first();

        if (! $tahunAjaran) {
            return [
                Stat::make('Error', 'Tahun Ajaran Aktif tidak ditemukan')->color('danger'),
            ];
        }

        // Global scope will automatically filter by madrasah
        $jenisPresensi = JenisPresensiMadrasah::where('is_active', true)->get();
        $stats = [];

        foreach ($jenisPresensi as $jenis) {
            $presensi = PresensiMadrasah::where('tanggal', $today)
                ->where('kelas_id', $kelasId)
                ->where('jenis_presensi_madrasah_id', $jenis->id)
                ->first();

            $status = $presensi ? 'Sudah Input' : 'Belum Input';
            $color = $presensi ? 'success' : 'danger';
            $icon = $presensi ? 'heroicon-o-check-circle' : 'heroicon-o-x-circle';
            $description = $presensi ? 'Terkunci: '.($presensi->is_locked ? 'Ya' : 'Tidak') : 'Klik untuk input';

            $url = \App\Filament\Pages\InputPresensiMadrasah::getUrl([
                'kelas_id' => $kelasId,
                'jenis_presensi_id' => $jenis->id,
                'tanggal' => $today->toDateString(),
            ]);

            $stat = Stat::make($jenis->nama, $status)
                ->description($description)
                ->descriptionIcon($icon)
                ->color($color)
                ->url($url);

            $stats[] = $stat;
        }

        return $stats;
    }
}
