<?php

namespace App\Filament\Widgets;

use App\Models\JenisPresensiAsrama;
use App\Models\Kamar;
use App\Models\PresensiAsrama;
use App\Models\TahunAjaran;
use BezhanSalleh\FilamentShield\Traits\HasWidgetShield;
use Carbon\Carbon;
use Filament\Widgets\Concerns\InteractsWithPageFilters;
use Filament\Widgets\StatsOverviewWidget as BaseWidget;
use Filament\Widgets\StatsOverviewWidget\Stat;
use Illuminate\Support\Facades\Auth;

class PresensiAsramaWidget extends BaseWidget
{
    use \App\Traits\HasStrictWidgetShield, InteractsWithPageFilters;

    protected static ?int $sort = 8;

    protected function getStats(): array
    {
        $user = Auth::user();
        $kamarId = null;

        // If user is a wali kamar, automatically find their kamar
        if ($user->hasRole('wali kamar')) {
            // Use global scope - it will automatically filter by madrasah
            $kamar = Kamar::where('wali_kamar_id', $user->id)->first();
            if ($kamar) {
                $kamarId = $kamar->id;
            }
        } else {
            // For other roles, use the page filter or the first kamar as a default
            // Global scope will automatically filter by madrasah
            $kamarId = $this->filters['kamar_id'] ?? Kamar::first()?->id;
        }

        if (! $kamarId) {
            // If user is wali kamar but no kamar is assigned, or no kamars exist for other roles
            $message = $user->hasRole('wali kamar') ? 'Anda tidak ditugaskan di kamar manapun.' : 'Belum ada data kamar.';

            return [
                Stat::make('Info', $message)
                    ->color('gray'),
            ];
        }

        // The rest of the original logic
        $today = Carbon::today();
        $tahunAjaran = TahunAjaran::where('is_aktif', true)->first();

        if (! $tahunAjaran) {
            return [
                Stat::make('Error', 'Tahun Ajaran Aktif tidak ditemukan')->color('danger'),
            ];
        }

        // Global scope will automatically filter by madrasah
        $jenisPresensi = JenisPresensiAsrama::where('is_active', true)->get();
        $stats = [];

        foreach ($jenisPresensi as $jenis) {
            $presensi = PresensiAsrama::where('tanggal', $today)
                ->where('kamar_id', $kamarId)
                ->where('jenis_presensi_asrama_id', $jenis->id)
                ->first();

            $status = $presensi ? 'Sudah Input' : 'Belum Input';
            $color = $presensi ? 'success' : 'danger';
            $icon = $presensi ? 'heroicon-o-check-circle' : 'heroicon-o-x-circle';
            $description = $presensi ? 'Terkunci: '.($presensi->is_locked ? 'Ya' : 'Tidak') : 'Klik untuk input';

            $url = route('filament.admin.pages.input-presensi-asrama', [
                'kamar_id' => $kamarId,
                'jenis_presensi_id' => $jenis->id,
                'tanggal' => $today->toDateString(),
            ]);

            $stats[] = Stat::make($jenis->nama, $status)
                ->description($description)
                ->descriptionIcon($icon)
                ->color($color)
                ->url($url);
        }

        return $stats;
    }
}
