<?php

namespace App\Filament\Resources;

use App\Filament\Exports\UserExporter;
use App\Filament\Resources\UserResource\Pages;
use App\Models\User;
use App\Imports\UserImport;
use App\Exports\UserTemplateExport;
use Filament\Forms;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Infolists\Components\Section as InfolistSection;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Infolist;
use Filament\Notifications\Notification;
use Filament\Resources\Resource;
use Filament\Support\Enums\FontWeight;
use Filament\Tables;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ExportAction;
use Filament\Tables\Actions\ExportBulkAction;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Table;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;
use STS\FilamentImpersonate\Tables\Actions\Impersonate;

class UserResource extends Resource
{
    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::count();
    }

    protected static ?string $model = User::class;

    protected static ?string $navigationIcon = 'heroicon-o-users';

    protected static ?string $pluralModelLabel = 'User';

    protected static ?string $navigationGroup = 'Settings';

    protected static ?int $navigationSort = 2;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make(
                    'User Information'
                )->schema([
                    TextInput::make('name')
                        ->required(),
                    TextInput::make('email')
                        ->required(),
                    TextInput::make('password')
                        ->required(),
                    Select::make('madrasah_id')
                        ->relationship('madrasah', 'nama')
                        ->searchable()
                        ->preload()
                        ->label('Madrasah')
                        ->helperText('Pilih madrasah untuk user ini (opsional untuk super admin/pimpinan)'),
                    Select::make('roles')
                        ->relationship('roles', 'name')
                        ->multiple()
                        ->searchable()
                        ->preload()
                        ->label('Role')
                        ->reactive()
                        ->afterStateUpdated(fn ($state, callable $set) => $set('siswa_id', null)),
                    Select::make('siswa_id')
                        ->label('Link ke Siswa (untuk Ortu)')
                        ->options(function ($get, $record) {
                            $query = \App\Models\Siswa::query();

                            // Filter by madrasah if selected
                            if ($madrasahId = $get('madrasah_id')) {
                                $query->whereHas('kelas', fn ($q) => $q->where('madrasah_id', $madrasahId));
                            }

                            // Only show siswa without user_id OR current user's siswa (for edit)
                            $query->where(function ($q) use ($record) {
                                $q->whereNull('user_id');
                                if ($record) {
                                    $q->orWhere('user_id', $record->id);
                                }
                            });

                            return $query->pluck('nama', 'id');
                        })
                        ->searchable()
                        ->helperText('Hanya siswa yang belum ter-link ke user lain yang akan muncul')
                        ->visible(fn ($get) => in_array('ortu', $get('roles') ?? []))
                        ->reactive()
                        ->afterStateUpdated(function ($state, $get, $set) {
                            if ($state) {
                                $siswa = \App\Models\Siswa::find($state);
                                if ($siswa && ! $get('madrasah_id')) {
                                    $set('madrasah_id', $siswa->kelas->madrasah_id);
                                }
                            }
                        }),
                ]),
            ]);
    }

    public static function canCreate(): bool
    {
        return true;
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\Layout\Split::make([
                    Tables\Columns\ImageColumn::make('avatar_url')
                        ->searchable()
                        ->circular()
                        ->grow(false)
                        ->getStateUsing(fn ($record) => $record->avatar_url
                            ? $record->avatar_url
                            : 'https://ui-avatars.com/api/?name='.urlencode($record->name)),
                    Tables\Columns\TextColumn::make('name')
                        ->searchable()
                        ->weight(FontWeight::Bold),
                    Tables\Columns\Layout\Stack::make([
                        Tables\Columns\TextColumn::make('roles.name')
                            ->searchable()
                            ->icon('heroicon-o-shield-check')
                            ->grow(false),
                        Tables\Columns\TextColumn::make('email')
                            ->icon('heroicon-m-envelope')
                            ->searchable()
                            ->grow(false),
                    ])->alignStart()->visibleFrom('lg')->space(1),
                ]),
            ])
            ->filters([
                //
                SelectFilter::make('roles')
                    ->relationship('roles', 'name')
                    ->multiple()
                    ->preload(),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Action::make('Set Role')
                    ->icon('heroicon-m-adjustments-vertical')
                    ->form([
                        Select::make('role')
                            ->relationship('roles', 'name')
                            ->multiple()
                            ->required()
                            ->searchable()
                            ->preload()
                            ->optionsLimit(10)
                            ->getOptionLabelFromRecordUsing(fn ($record) => $record->name),
                    ]),
                // Impersonate::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->headerActions([
                ExportAction::make()
                    ->exporter(UserExporter::class),
                Action::make('import')
                    ->label('Import User')
                    ->icon('heroicon-o-arrow-up-tray')
                    ->color('primary')
                    ->form([
                        Forms\Components\FileUpload::make('file')
                            ->label('File Excel')
                            ->acceptedFileTypes([
                                'application/vnd.ms-excel',
                                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                            ])
                            ->required()
                            ->helperText('Upload file Excel (.xlsx atau .xls)'),
                        Forms\Components\Actions::make([
                            Forms\Components\Actions\Action::make('downloadTemplate')
                                ->label('Download Template Excel')
                                ->icon('heroicon-o-arrow-down-tray')
                                ->color('success')
                                ->action(function () {
                                    $filename = 'template-import-user.xlsx';
                                    return Excel::download(
                                        new UserTemplateExport(),
                                        $filename
                                    );
                                }),
                        ])->fullWidth(),
                    ])
                    ->action(function (array $data) {
                        try {
                            $import = new UserImport();
                            $fileKey = $data['file'];
                            
                            // Get file path logic similar to SiswaResource
                            $possiblePaths = [
                                storage_path('app/livewire-tmp/'.$fileKey),
                                storage_path('app/public/'.$fileKey),
                                storage_path('app/'.$fileKey),
                            ];

                            $filePath = null;
                            foreach ($possiblePaths as $path) {
                                if (file_exists($path)) {
                                    $filePath = $path;
                                    break;
                                }
                            }

                            if (! $filePath) {
                                $livewireTmpDir = storage_path('app/livewire-tmp');
                                if (is_dir($livewireTmpDir)) {
                                    $iterator = new \RecursiveIteratorIterator(
                                        new \RecursiveDirectoryIterator($livewireTmpDir, \RecursiveDirectoryIterator::SKIP_DOTS)
                                    );
                                    foreach ($iterator as $file) {
                                        if ($file->isFile() && (str_contains($file->getFilename(), $fileKey))) {
                                            $filePath = $file->getPathname();
                                            break;
                                        }
                                    }
                                }
                            }

                            if (! $filePath || ! file_exists($filePath)) {
                                throw new \Exception('File tidak ditemukan.');
                            }

                            Excel::import($import, $filePath);

                            if (file_exists($filePath)) {
                                @unlink($filePath);
                            }

                            $successCount = $import->getSuccessCount();
                            $failedCount = $import->getFailedCount();
                            $errors = $import->getErrors();

                            if ($failedCount > 0) {
                                $errorMessage = "Berhasil: {$successCount}, Gagal: {$failedCount}\n\n";
                                $errorMessage .= "Detail Error:\n".implode("\n", array_slice($errors, 0, 10));

                                Notification::make()
                                    ->title('Import selesai dengan beberapa error')
                                    ->body($errorMessage)
                                    ->warning()
                                    ->duration(10000)
                                    ->send();
                            } else {
                                Notification::make()
                                    ->title('Import berhasil!')
                                    ->body("Berhasil mengimport {$successCount} user")
                                    ->success()
                                    ->send();
                            }
                        } catch (\Exception $e) {
                            Notification::make()
                                ->title('Import gagal!')
                                ->body($e->getMessage())
                                ->danger()
                                ->send();
                        }
                    })
                    ->modalWidth('lg')
                    ->modalHeading('Import Data User')
                    ->modalDescription('Upload file Excel untuk mengimport data user.')
                    ->modalSubmitActionLabel('Import')
                    ->modalCancelActionLabel('Batal'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
                ExportBulkAction::make()
                    ->exporter(UserExporter::class),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListUsers::route('/'),
            'create' => Pages\CreateUser::route('/create'),
            // 'view' => Pages\ViewUser::route('/{record}'),
            'edit' => Pages\EditUser::route('/{record}/edit'),
        ];
    }

    public static function infolist(Infolist $infolist): Infolist
    {
        return $infolist
            ->schema([
                InfolistSection::make('User Information')->schema([
                    TextEntry::make('name'),
                    TextEntry::make('email'),
                ]),
            ]);
    }
}
