<?php

namespace App\Filament\Resources\PresensiNonAsramaResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;

class DetailPresensiNonAsramasRelationManager extends RelationManager
{
    protected static string $relationship = 'detailPresensiNonAsramas';
    
    protected static ?string $title = 'Presensi Siswa';

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Select::make('status')
                    ->options([
                        'hadir' => 'Hadir',
                        'sakit' => 'Sakit',
                        'izin' => 'Izin',
                        'alpha' => 'Alpha',
                    ])
                    ->required()
                    ->label('Status Presensi'),
                Forms\Components\Textarea::make('keterangan')
                    ->label('Keterangan')
                    ->rows(2)
                    ->maxLength(255),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('status')
            ->modifyQueryUsing(function (Builder $query) {
                $user = \Illuminate\Support\Facades\Auth::user();
                if ($user->hasRole(['ortu', 'Ortu'])) {
                     $siswa = $user->siswa;
                     if ($siswa) {
                          $query->where('siswa_id', $siswa->id);
                     } else {
                          $query->whereRaw('1 = 0');
                     }
                }
            })
            ->columns([
                Tables\Columns\TextColumn::make('siswa.nama')
                    ->label('Nama Siswa')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('siswa.nisn')
                    ->label('NISN')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('status')
                    ->badge()
                    ->color(fn (string $state): string => match ($state) {
                        'hadir' => 'success',
                        'sakit' => 'warning',
                        'izin' => 'info',
                        'alpha' => 'danger',
                        default => 'gray',
                    })
                    ->label('Status'),
                Tables\Columns\TextColumn::make('keterangan')
                    ->label('Keterangan')
                    ->limit(50),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('status')
                    ->options([
                        'hadir' => 'Hadir',
                        'sakit' => 'Sakit',
                        'izin' => 'Izin',
                        'alpha' => 'Alpha',
                    ]),
            ])
       
            ->actions([
                Tables\Actions\EditAction::make(),
                // Tables\Actions\DeleteAction::make(), // Should not delete, just edit
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\BulkAction::make('update_status')
                        ->label('Update Status')
                        ->icon('heroicon-o-pencil-square')
                        ->hidden(fn () => Auth::user()->hasRole('ortu')) // Ortu cannot bulk update others
                        ->form([
                            Forms\Components\Select::make('status')
                                ->options([
                                    'hadir' => 'Hadir',
                                    'sakit' => 'Sakit',
                                    'izin' => 'Izin',
                                    'alpha' => 'Alpha',
                                ])
                                ->required(),
                            Forms\Components\Textarea::make('keterangan')
                                ->label('Keterangan (Opsional)')
                                ->rows(2),
                        ])
                        ->action(function (Table $table, array $data) {
                            $records = $table->getSelectedRecords();
                            foreach ($records as $record) {
                                $updateData = ['status' => $data['status']];
                                if (!empty($data['keterangan'])) {
                                    $updateData['keterangan'] = $data['keterangan'];
                                }
                                $record->update($updateData);
                            }
                        })
                        ->deselectRecordsAfterCompletion(),
                ]),
            ]);
    }
}
