<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PresensiNonAsramaResource\Pages;
use App\Filament\Resources\PresensiNonAsramaResource\RelationManagers;
use App\Models\PresensiNonAsrama;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Support\Facades\Auth;

class PresensiNonAsramaResource extends Resource
{
    protected static ?string $model = PresensiNonAsrama::class;

    protected static ?string $navigationIcon = 'heroicon-o-clipboard-document-list';

    protected static ?string $pluralModelLabel = 'Presensi Non Asrama';

    protected static ?string $navigationGroup = 'Kegiatan Pengasuhan';

    protected static ?int $navigationSort = 42;

    public static function getEloquentQuery(): \Illuminate\Database\Eloquent\Builder
    {
        $query = parent::getEloquentQuery();
        $user = Auth::user();

        if (! $user) {
            return $query;
        }

        // Super admin, pimpinan, operator can see all
        if ($user->hasRole(['super_admin', 'pimpinan', 'operator'])) {
            return $query;
        }

        // Get user's madrasah ID
        $madrasahId = $user->getMadrasahId();

        if (! $madrasahId) {
            return $query->whereRaw('1 = 0');
        }

        // Filter through kelas->madrasah relationship
        $query->whereHas('kelas', function ($q) use ($madrasahId) {
            $q->where('madrasah_id', $madrasahId);
        });

        // Walikelas hanya bisa lihat presensi kelas yang dia pegang
        if ($user->hasRole('walikelas')) {
            $kelas = \App\Models\Kelas::where('wali_kelas_id', $user->id)->first();
            if ($kelas) {
                $query->where('kelas_id', $kelas->id);
            } else {
                $query->whereRaw('1 = 0');
            }
        }

        // Ortu hanya bisa lihat presensi dimana anaknya berada
        if ($user->hasRole(['ortu', 'Ortu'])) {
            $siswa = $user->siswa;
            if ($siswa) {
                $query->where('kelas_id', $siswa->kelas_id)
                    ->whereHas('detailPresensiNonAsramas', function ($q) use ($siswa) {
                        $q->where('siswa_id', $siswa->id);
                    });
            } else {
                $query->whereRaw('1 = 0');
            }
        }

        return $query;
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('Informasi Presensi')
                    ->schema([
                        Forms\Components\DatePicker::make('tanggal')
                            ->required()
                            ->default(now())
                            ->displayFormat('d/m/Y')
                            ->label('Tanggal'),
                        Forms\Components\Select::make('tahun_ajaran_id')
                            ->relationship('tahunAjaran', 'nama')
                            ->required()
                            ->searchable()
                            ->preload()
                            ->label('Tahun Ajaran'),
                        Forms\Components\Select::make('jenis_presensi_non_asrama_id')
                            ->relationship('jenisPresensiNonAsrama', 'nama')
                            ->required()
                            ->searchable()
                            ->preload()
                            ->label('Jenis Presensi'),
                        Forms\Components\Select::make('kelas_id')
                            ->relationship('kelas', 'nama', function ($query) {
                                $user = Auth::user();
                                if ($user && $user->hasRole('walikelas')) {
                                    $query->where('wali_kelas_id', $user->id);
                                }

                                return $query;
                            })
                            ->required()
                            ->searchable()
                            ->preload()
                            ->label('Kelas'),
                        Forms\Components\Select::make('user_id')
                            ->relationship('user', 'name')
                            ->required()
                            ->searchable()
                            ->preload()
                            ->default(Auth::id())
                            ->label('Petugas Input'),

                        Forms\Components\Toggle::make('is_locked')
                            ->label('Kunci Presensi')
                            ->helperText('Jika dikunci, presensi tidak bisa diedit lagi')
                            ->default(false),
                        Forms\Components\Textarea::make('keterangan')
                            ->rows(3)
                            ->columnSpanFull()
                            ->label('Keterangan'),
                    ])
                    ->columns(2),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->defaultSort('tanggal', 'desc')
            ->columns([
                Tables\Columns\TextColumn::make('tanggal')
                    ->date('d/m/Y')
                    ->sortable()
                    ->label('Tanggal'),
                Tables\Columns\TextColumn::make('kelas.nama')
                    ->searchable()
                    ->sortable()
                    ->label('Kelas'),
                Tables\Columns\TextColumn::make('jenisPresensiNonAsrama.nama')
                    ->searchable()
                    ->sortable()
                    ->label('Jenis Presensi'),
                Tables\Columns\TextColumn::make('jenisPresensiNonAsrama.kategori')
                    ->badge()
                    ->label('Kategori'),
                Tables\Columns\TextColumn::make('detail_presensi_non_asramas_count')
                    ->counts('detailPresensiNonAsramas')
                    ->label('Jml Siswa')
                    ->badge()
                    ->color('success'),
                Tables\Columns\IconColumn::make('is_locked')
                    ->boolean()
                    ->label('Terkunci')
                    ->trueIcon('heroicon-o-lock-closed')
                    ->falseIcon('heroicon-o-lock-open'),
                Tables\Columns\TextColumn::make('user.name')
                    ->label('Petugas')
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->label('Dibuat'),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('tahun_ajaran_id')
                    ->relationship('tahunAjaran', 'nama')
                    ->label('Tahun Ajaran'),
                Tables\Filters\SelectFilter::make('kelas_id')
                    ->relationship('kelas', 'nama', function ($query) {
                        $user = Auth::user();
                        if ($user && $user->hasRole('walikelas')) {
                            $query->where('wali_kelas_id', $user->id);
                        }

                        return $query;
                    })
                    ->label('Kelas'),
                Tables\Filters\SelectFilter::make('jenis_presensi_non_asrama_id')
                    ->relationship('jenisPresensiNonAsrama', 'nama')
                    ->label('Jenis Presensi'),
                Tables\Filters\TernaryFilter::make('is_locked')
                    ->label('Status Kunci'),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make()
                    ->visible(fn ($record) => ! $record->is_locked),
                Tables\Actions\Action::make('toggle_lock')
                    ->label(fn ($record) => $record->is_locked ? 'Buka Kunci' : 'Kunci')
                    ->icon(fn ($record) => $record->is_locked ? 'heroicon-o-lock-open' : 'heroicon-o-lock-closed')
                    ->color(fn ($record) => $record->is_locked ? 'warning' : 'success')
                    ->requiresConfirmation()
                    ->action(fn ($record) => $record->update(['is_locked' => ! $record->is_locked])),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            RelationManagers\DetailPresensiNonAsramasRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPresensiNonAsramas::route('/'),
            'create' => Pages\CreatePresensiNonAsrama::route('/create'),
            'edit' => Pages\EditPresensiNonAsrama::route('/{record}/edit'),
        ];
    }
}
