<?php

namespace App\Filament\Resources\PresensiMadrasahResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Collection;

class DetailPresensiMadrasahsRelationManager extends RelationManager
{
    protected static string $relationship = 'detailPresensiMadrasahs';

    protected static ?string $title = 'Detail Presensi Siswa';

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Select::make('siswa_id')
                    ->relationship('siswa', 'nama')
                    ->disabled()
                    ->required()
                    ->label('Siswa'),
                Forms\Components\Select::make('status')
                    ->options([
                        'hadir' => 'Hadir',
                        'sakit' => 'Sakit',
                        'izin' => 'Izin',
                        'terlambat' => 'Terlambat',
                        'cabut' => 'Cabut',
                        'alpha' => 'Alpha',
                    ])
                    ->required()
                    ->default('hadir')
                    ->label('Status Kehadiran'),
                Forms\Components\Textarea::make('keterangan')
                    ->rows(2)
                    ->columnSpanFull(),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('status')
            ->columns([
                Tables\Columns\TextColumn::make('siswa.nisn')
                    ->label('NISN')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('siswa.nama')
                    ->label('Nama Siswa')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('status')
                    ->badge()
                    ->color(fn (?string $state): string => match ($state) {
                        'hadir' => 'success',
                        'sakit' => 'warning',
                        'izin' => 'info',
                        'alpha' => 'danger',
                        default => 'gray',
                    })
                    ->formatStateUsing(fn (?string $state): string => $state ? ucfirst($state) : '-')
                    ->sortable(),
                Tables\Columns\TextColumn::make('keterangan')
                    ->limit(30)
                    ->tooltip(fn (?string $state): ?string => $state),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('status')
                    ->options([
                        'hadir' => 'Hadir',
                        'sakit' => 'Sakit',
                        'izin' => 'Izin',
                        'alpha' => 'Alpha',
                    ]),
            ])
            ->headerActions([
                // Tables\Actions\CreateAction::make()
                //     ->label('Tambah Manual')
                //     ->visible(fn ($livewire) => ! $livewire->getOwnerRecord()->is_locked),
                Tables\Actions\Action::make('generate_siswa')
                    ->label('Buat Absen')
                    ->icon('heroicon-o-users')
                    ->requiresConfirmation()
                    ->color('primary')
                    ->action(function ($livewire) {
                        $presensi = $livewire->getOwnerRecord();
                        $kelas = $presensi->kelas;

                        if (! $kelas) {
                            Notification::make()
                                ->title('Kelas tidak ditemukan')
                                ->danger()
                                ->send();

                            return;
                        }

                        $siswas = $kelas->siswas;
                        $count = 0;

                        foreach ($siswas as $siswa) {
                            // Check if already exists
                            $exists = $presensi->detailPresensiMadrasahs()
                                ->where('siswa_id', $siswa->id)
                                ->exists();

                            if (! $exists) {
                                $presensi->detailPresensiMadrasahs()->create([
                                    'siswa_id' => $siswa->id,
                                    'status' => 'hadir', // Default
                                ]);
                                $count++;
                            }
                        }

                        Notification::make()
                            ->title("Berhasil generate $count siswa")
                            ->success()
                            ->send();
                    })
                    ->visible(fn ($livewire) => $livewire->getOwnerRecord()->detailPresensiMadrasahs()->count() === 0 && ! $livewire->getOwnerRecord()->is_locked),
            ])
            ->actions([
                Tables\Actions\EditAction::make()
                    ->visible(fn ($livewire) => ! $livewire->getOwnerRecord()->is_locked),
                Tables\Actions\DeleteAction::make()
                    ->visible(fn ($livewire) => ! $livewire->getOwnerRecord()->is_locked),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make()
                        ->visible(fn ($livewire) => ! $livewire->getOwnerRecord()->is_locked),
                    Tables\Actions\BulkAction::make('set_status')
                        ->label('Set Status')
                        ->icon('heroicon-o-pencil')
                        ->form([
                            Forms\Components\Select::make('status')
                                ->options([
                                    'hadir' => 'Hadir',
                                    'sakit' => 'Sakit',
                                    'izin' => 'Izin',
                                    'alpha' => 'Alpha',
                                ])
                                ->required(),
                        ])
                        ->action(function (Collection $records, array $data) {
                            $records->each(function ($record) use ($data) {
                                $record->update(['status' => $data['status']]);
                            });

                            Notification::make()
                                ->title('Status berhasil diperbarui')
                                ->success()
                                ->send();
                        })
                        ->deselectRecordsAfterCompletion()
                        ->visible(fn ($livewire) => ! $livewire->getOwnerRecord()->is_locked),
                ]),
            ]);
    }
}
