<?php

namespace App\Filament\Resources;

use App\Filament\Resources\LaporanPresensiMadrasahResource\Pages;
use App\Models\DetailPresensiMadrasah;
use App\Models\JenisPresensiMadrasah;
use App\Models\Kelas;
use App\Models\LaporanPresensiMadrasah;
use App\Models\Siswa;
use App\Models\TahunAjaran;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;

class LaporanPresensiMadrasahResource extends Resource
{
    protected static ?string $model = LaporanPresensiMadrasah::class;

    protected static ?string $navigationIcon = 'heroicon-o-document-chart-bar';

    protected static ?string $navigationLabel = ' Madrasah';

    protected static ?string $pluralModelLabel = ' Presensi Madrasah';

    protected static ?string $navigationGroup = 'Laporan';

    protected static ?int $navigationSort = 50;

    public static function form(Form $form): Form
    {
        return $form->schema([]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->query(function () {
                $query = Siswa::query()->with(['kelas']);

                // Scoped access untuk wali kelas - hanya lihat siswa di kelasnya
                $user = Auth::user();

                if ($user) {
                    // Super admin, pimpinan, operator can see all
                    if (! $user->hasRole(['super_admin', 'pimpinan', 'operator'])) {
                        // Get user's madrasah ID
                        $madrasahId = $user->getMadrasahId();

                        if ($madrasahId) {
                            // Filter through kelas->madrasah relationship
                            $query->whereHas('kelas', function ($q) use ($madrasahId) {
                                $q->where('madrasah_id', $madrasahId);
                            });
                        } else {
                            // If user has no madrasah, return empty query
                            $query->whereRaw('1 = 0');
                        }
                    }

                    // Walikelas only sees students in their class
                    if ($user->hasRole('walikelas')) {
                        $kelas = \App\Models\Kelas::where('wali_kelas_id', $user->id)->first();
                        if ($kelas) {
                            $query->where('kelas_id', $kelas->id);
                        } else {
                            // Jika wali kelas tidak punya kelas, return empty query
                            $query->whereRaw('1 = 0');
                        }
                    }
                }

                $filters = [
                    'tahun_ajaran_id' => request()->input('tableFilters.tahun_ajaran_id.value'),
                    'jenis_presensi_madrasah_id' => request()->input('tableFilters.jenis_presensi_madrasah_id.value'),
                    'dari' => request()->input('tableFilters.tanggal.dari'),
                    'sampai' => request()->input('tableFilters.tanggal.sampai'),
                ];

                $filterCallback = function ($q) use ($filters) {
                    $q->whereHas('presensiMadrasah', function ($q) use ($filters) {
                        if ($filters['tahun_ajaran_id']) {
                            $q->where('tahun_ajaran_id', $filters['tahun_ajaran_id']);
                        }
                        if ($filters['jenis_presensi_madrasah_id']) {
                            $q->where('jenis_presensi_madrasah_id', $filters['jenis_presensi_madrasah_id']);
                        }
                        if ($filters['dari']) {
                            $q->where('tanggal', '>=', $filters['dari']);
                        }
                        if ($filters['sampai']) {
                            $q->where('tanggal', '<=', $filters['sampai']);
                        }
                    });
                };

                $query->withCount([
                    'detailPresensiMadrasahs as total_hadir' => fn ($q) => $q->where('status', 'hadir')->tap($filterCallback),
                    'detailPresensiMadrasahs as total_sakit' => fn ($q) => $q->where('status', 'sakit')->tap($filterCallback),
                    'detailPresensiMadrasahs as total_izin' => fn ($q) => $q->where('status', 'izin')->tap($filterCallback),
                    'detailPresensiMadrasahs as total_alpha' => fn ($q) => $q->where('status', 'alpha')->tap($filterCallback),
                    'detailPresensiMadrasahs as cabut' => fn ($q) => $q->where('status', 'cabut')->tap($filterCallback),
                    'detailPresensiMadrasahs as terlambat' => fn ($q) => $q->where('status', 'terlambat')->tap($filterCallback),
                ]);

                return $query;
            })
            ->columns([
                Tables\Columns\TextColumn::make('nisn')
                    ->label('NISN')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('nama')
                    ->label('Nama Siswa')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('kelas.nama')
                    ->label('Kelas')
                    ->sortable(),
                Tables\Columns\TextColumn::make('total_hadir')
                    ->label('H')
                    ->alignCenter()
                    ->sortable(),
                Tables\Columns\TextColumn::make('total_sakit')
                    ->label('S')
                    ->alignCenter()
                    ->sortable(),
                Tables\Columns\TextColumn::make('total_izin')
                    ->label('I')
                    ->alignCenter()
                    ->sortable(),
                Tables\Columns\TextColumn::make('cabut')
                    ->label('C')
                    ->alignCenter()
                    ->sortable(),
                Tables\Columns\TextColumn::make('terlambat')
                    ->label('T')
                    ->alignCenter()
                    ->sortable(),
                Tables\Columns\TextColumn::make('total_alpha')
                    ->label('A')
                    ->alignCenter()
                    ->sortable(),
                Tables\Columns\TextColumn::make('persentase')
                    ->label('Persentase Hadir')
                    ->getStateUsing(function ($record) {
                        $total = $record->total_hadir + $record->total_sakit + $record->total_izin + $record->total_alpha;

                        return $total > 0 ? round(($record->total_hadir / $total) * 100, 2).'%' : '0%';
                    })
                    ->alignCenter()
                    ->badge()
                    ->color(function (string $state): string {
                        $percentage = (float) str_replace('%', '', $state);

                        return match (true) {
                            $percentage >= 90 => 'success',
                            $percentage >= 75 => 'warning',
                            default => 'danger',
                        };
                    }),
                Tables\Columns\TextColumn::make('keterangan')
                    ->label('Keterangan')
                    ->getStateUsing(function ($record) {
                        // Get filters
                        $filters = [
                            'tahun_ajaran_id' => request()->input('tableFilters.tahun_ajaran_id.value'),
                            'jenis_presensi_madrasah_id' => request()->input('tableFilters.jenis_presensi_madrasah_id.value'),
                            'dari' => request()->input('tableFilters.tanggal.dari'),
                            'sampai' => request()->input('tableFilters.tanggal.sampai'),
                        ];

                        // Get keterangan for non-hadir status
                        $query = DetailPresensiMadrasah::where('siswa_id', $record->id)
                            ->whereIn('status', ['sakit', 'izin', 'alpha', 'cabut', 'terlambat'])
                            ->whereNotNull('keterangan')
                            ->where('keterangan', '!=', '');

                        // Apply filters
                        $query->whereHas('presensiMadrasah', function ($q) use ($filters) {
                            if ($filters['tahun_ajaran_id']) {
                                $q->where('tahun_ajaran_id', $filters['tahun_ajaran_id']);
                            }
                            if ($filters['jenis_presensi_madrasah_id']) {
                                $q->where('jenis_presensi_madrasah_id', $filters['jenis_presensi_madrasah_id']);
                            }
                            if ($filters['dari']) {
                                $q->where('tanggal', '>=', $filters['dari']);
                            }
                            if ($filters['sampai']) {
                                $q->where('tanggal', '<=', $filters['sampai']);
                            }
                        });

                        $details = $query->with('presensiMadrasah:id,tanggal')
                            ->orderBy('created_at', 'desc')
                            ->get();

                        if ($details->isEmpty()) {
                            return '-';
                        }

                        // Group by status and create summary like PDF format
                        $grouped = [];
                        foreach ($details as $detail) {
                            $status = $detail->status;
                            if (! isset($grouped[$status])) {
                                $grouped[$status] = [];
                            }
                            $grouped[$status][] = $detail->keterangan;
                        }

                        $parts = [];
                        foreach ($grouped as $status => $keterangans) {
                            $statusLabel = strtoupper(substr($status, 0, 1)); // S, I, A, C, T
                            $count = count($keterangans);
                            // Show first keterangan as sample (30 chars)
                            $sample = mb_substr($keterangans[0], 0, 30);
                            if (mb_strlen($keterangans[0]) > 30) {
                                $sample .= '...';
                            }
                            $parts[] = "{$statusLabel}: {$sample}".($count > 1 ? ' (+'.($count - 1).')' : '');
                        }

                        return implode('; ', $parts);
                    })
                    ->searchable(false)
                    ->sortable(false)
                    ->wrap(),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('tahun_ajaran_id')
                    ->label('Tahun Ajaran')
                    ->options(TahunAjaran::pluck('nama', 'id'))
                    ->default(TahunAjaran::where('is_aktif', true)->first()?->id)
                    ->query(function (Builder $query, array $data): Builder {
                        // Filter akan ditangani di withCount
                        return $query;
                    })
                    ->searchable()
                    ->preload(),
                Tables\Filters\SelectFilter::make('madrasah')
                    ->label('Madrasah')
                    ->relationship('madrasah', 'nama'),
                Tables\Filters\SelectFilter::make('kelas_id')
                    ->label('Kelas')
                    ->options(function () {
                        $user = Auth::user();
                        $query = Kelas::query();

                        // Super admin bisa lihat semua kelas tanpa filter tingkat
                        if (! $user || ! $user->hasRole('super_admin')) {
                            // Filter hanya tingkat VII-XII untuk non super admin
                            $query->whereIn('tingkat', ['VII', 'VIII', 'IX', 'X', 'XI', 'XII']);
                        }

                        // Filter by User's Madrasah (kecuali super_admin, pimpinan, operator)
                        if ($user && ! $user->hasRole(['super_admin', 'pimpinan', 'operator'])) {
                            if ($madrasahId = $user->getMadrasahId()) {
                                $query->where('madrasah_id', $madrasahId);
                            }
                        }

                        // Filter by Wali Kelas
                        if ($user && $user->hasRole('walikelas')) {
                            $query->where('wali_kelas_id', $user->id);
                        }

                        // Sort by tingkat and nama
                        return $query->orderByRaw("FIELD(tingkat, 'VII', 'VIII', 'IX', 'X', 'XI', 'XII')")
                            ->orderBy('nama')
                            ->pluck('nama', 'id');
                    })
                    ->query(function (Builder $query, array $data): Builder {
                        if (! empty($data['value'])) {
                            $query->whereHas('kelas', fn ($q) => $q->where('id', $data['value']));
                        }

                        return $query;
                    })
                    ->searchable()
                    ->preload(),

                Tables\Filters\SelectFilter::make('jenis_presensi_madrasah_id')
                    ->label('Jenis Presensi')
                    ->options(JenisPresensiMadrasah::where('is_active', true)->pluck('nama', 'id'))
                    ->query(function (Builder $query, array $data): Builder {
                        // Filter akan ditangani di withCount
                        return $query;
                    })
                    ->searchable()
                    ->preload(),

                Tables\Filters\Filter::make('tanggal')
                    ->form([
                        \Filament\Forms\Components\DatePicker::make('dari')
                            ->label('Dari Tanggal')
                            ->default(now()->startOfMonth()),
                        \Filament\Forms\Components\DatePicker::make('sampai')
                            ->label('Sampai Tanggal')
                            ->default(now()),
                    ])
                    ->indicateUsing(function (array $data): array {
                        $indicators = [];
                        if ($data['dari'] ?? null) {
                            $indicators[] = 'Dari: '.\Carbon\Carbon::parse($data['dari'])->format('d/m/Y');
                        }
                        if ($data['sampai'] ?? null) {
                            $indicators[] = 'Sampai: '.\Carbon\Carbon::parse($data['sampai'])->format('d/m/Y');
                        }

                        return $indicators;
                    }),
            ])
            ->actions([
                Tables\Actions\Action::make('lihat_keterangan')
                    ->label('Detail')
                    ->icon('heroicon-o-eye')
                    ->color('info')
                    ->modalHeading(fn ($record) => 'Detail Keterangan - '.$record->nama)
                    ->modalDescription(fn ($record) => 'NISN: '.$record->nisn.' | Kelas: '.($record->kelas->nama ?? '-'))
                    ->modalContent(function ($record) {
                        // Get filters
                        $filters = [
                            'tahun_ajaran_id' => request()->input('tableFilters.tahun_ajaran_id.value'),
                            'jenis_presensi_madrasah_id' => request()->input('tableFilters.jenis_presensi_madrasah_id.value'),
                            'dari' => request()->input('tableFilters.tanggal.dari'),
                            'sampai' => request()->input('tableFilters.tanggal.sampai'),
                        ];

                        // Get all keterangan
                        $query = DetailPresensiMadrasah::where('siswa_id', $record->id)
                            ->whereIn('status', ['sakit', 'izin', 'alpha', 'cabut', 'terlambat'])
                            ->whereNotNull('keterangan')
                            ->where('keterangan', '!=', '');

                        $query->whereHas('presensiMadrasah', function ($q) use ($filters) {
                            if ($filters['tahun_ajaran_id']) {
                                $q->where('tahun_ajaran_id', $filters['tahun_ajaran_id']);
                            }
                            if ($filters['jenis_presensi_madrasah_id']) {
                                $q->where('jenis_presensi_madrasah_id', $filters['jenis_presensi_madrasah_id']);
                            }
                            if ($filters['dari']) {
                                $q->where('tanggal', '>=', $filters['dari']);
                            }
                            if ($filters['sampai']) {
                                $q->where('tanggal', '<=', $filters['sampai']);
                            }
                        });

                        $details = $query->with(['presensiMadrasah' => function ($q) {
                            $q->select('id', 'tanggal', 'jenis_presensi_madrasah_id')
                                ->with('jenisPresensiMadrasah:id,nama');
                        }])
                            ->orderBy('created_at', 'desc')
                            ->get();

                        if ($details->isEmpty()) {
                            return view('filament.components.empty-state', [
                                'message' => 'Tidak ada keterangan untuk siswa ini.',
                            ]);
                        }

                        return view('filament.components.keterangan-detail', [
                            'details' => $details,
                        ]);
                    })
                    ->modalSubmitAction(false)
                    ->modalCancelActionLabel('Tutup')
                    ->visible(fn ($record) => DetailPresensiMadrasah::where('siswa_id', $record->id)
                        ->whereIn('status', ['sakit', 'izin', 'alpha', 'cabut', 'terlambat'])
                        ->whereNotNull('keterangan')
                        ->where('keterangan', '!=', '')
                        ->exists()
                    ),
            ])
            ->recordAction('lihat_keterangan')
            ->recordUrl(null)
            ->defaultSort('nama');
    }

    protected static function getAttendanceCount($siswa, $status, $table): int
    {
        $filters = $table->getFilters();

        $query = DetailPresensiMadrasah::where('siswa_id', $siswa->id)
            ->where('status', $status);

        // Apply filters
        if ($tahunAjaranId = request()->input('tableFilters.tahun_ajaran_id.value')) {
            $query->whereHas('presensiMadrasah', fn ($q) => $q->where('tahun_ajaran_id', $tahunAjaranId));
        }

        if ($jenisPresensiId = request()->input('tableFilters.jenis_presensi_madrasah_id.value')) {
            $query->whereHas('presensiMadrasah', fn ($q) => $q->where('jenis_presensi_madrasah_id', $jenisPresensiId));
        }

        if ($dari = request()->input('tableFilters.tanggal.dari')) {
            $query->whereHas('presensiMadrasah', fn ($q) => $q->where('tanggal', '>=', $dari));
        }

        if ($sampai = request()->input('tableFilters.tanggal.sampai')) {
            $query->whereHas('presensiMadrasah', fn ($q) => $q->where('tanggal', '<=', $sampai));
        }

        return $query->count();
    }

    protected static function getTotalAttendance($siswa, $table): int
    {
        $query = DetailPresensiMadrasah::where('siswa_id', $siswa->id);

        // Apply same filters
        if ($tahunAjaranId = request()->input('tableFilters.tahun_ajaran_id.value')) {
            $query->whereHas('presensiMadrasah', fn ($q) => $q->where('tahun_ajaran_id', $tahunAjaranId));
        }

        if ($jenisPresensiId = request()->input('tableFilters.jenis_presensi_madrasah_id.value')) {
            $query->whereHas('presensiMadrasah', fn ($q) => $q->where('jenis_presensi_madrasah_id', $jenisPresensiId));
        }

        if ($dari = request()->input('tableFilters.tanggal.dari')) {
            $query->whereHas('presensiMadrasah', fn ($q) => $q->where('tanggal', '>=', $dari));
        }

        if ($sampai = request()->input('tableFilters.tanggal.sampai')) {
            $query->whereHas('presensiMadrasah', fn ($q) => $q->where('tanggal', '<=', $sampai));
        }

        return $query->count();
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListLaporanPresensiMadrasahs::route('/'),
        ];
    }
}
