<?php

namespace App\Filament\Resources\KelasResource\RelationManagers;

use App\Models\Kelas;
use App\Models\RiwayatKelas;
use App\Models\Siswa;
use App\Models\TahunAjaran;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class SiswasRelationManager extends RelationManager
{
    protected static string $relationship = 'siswas';

    public static function canViewForRecord($ownerRecord, $panel): bool
    {
        $user = Auth::user();
        if (! $user) {
            return false;
        }

        if ($user->hasRole('walikelas')) {
            return (int) ($ownerRecord->wali_kelas_id ?? 0) === (int) $user->id;
        }

        return true;
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('nama')
                    ->required()
                    ->maxLength(255),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->heading('Siswa')
            ->description('Manage Siswa disini.')
            ->recordTitleAttribute('nama')
            ->modifyQueryUsing(fn ($query) => $query->aktif()) // Only show active students
            ->columns([
                Tables\Columns\TextColumn::make('nisn')->searchable(),
                Tables\Columns\TextColumn::make('nama')->searchable(),
                Tables\Columns\TextColumn::make('jenis_kelamin'),
            ])
            ->filters([
                //
            ])
            ->headerActions([
                // Tables\Actions\CreateAction::make(),
                Tables\Actions\BulkAction::make('naik_kelas')
                    ->icon('heroicon-o-arrow-up-on-square')
                    ->label('Naikkan Kelas')
                    ->form(function () {
                        $currentClass = $this->getOwnerRecord();
                        $currentLevel = $currentClass->tingkat;
                        $scenario = $this->detectGraduationScenario($currentLevel);

                        // Build form based on scenario
                        return $this->buildFormForScenario($scenario, $currentClass);
                    })
                    ->action(function (Collection $records, array $data) {
                        try {
                            // Validate Tahun Ajaran
                            $activeTahunAjaran = TahunAjaran::where('is_aktif', true)->first();

                            if (! $activeTahunAjaran) {
                                Notification::make()
                                    ->title('Aksi Gagal')
                                    ->body('Tidak ada Tahun Ajaran yang aktif. Silakan aktifkan satu terlebih dahulu.')
                                    ->danger()
                                    ->send();

                                return;
                            }

                            // Get current class and detect scenario
                            $currentClass = $this->getOwnerRecord();
                            $currentLevel = $currentClass->tingkat;
                            $scenario = $this->detectGraduationScenario($currentLevel);

                            // Route to appropriate handler
                            if ($scenario === 'normal') {
                                $this->handleNormalPromotion($records, $data, $activeTahunAjaran);
                            } elseif ($scenario === 'graduation_mts') {
                                $this->handleGraduationMTs($records, $data, $activeTahunAjaran);
                            } elseif ($scenario === 'graduation_ma') {
                                $this->handleGraduationMA($records, $data, $activeTahunAjaran);
                            }

                        } catch (\Exception $e) {
                            // Show detailed error notification for debugging
                            Notification::make()
                                ->title('Terjadi Kesalahan')
                                ->body('Error: '.$e->getMessage().' | File: '.basename($e->getFile()).' | Line: '.$e->getLine())
                                ->danger()
                                ->persistent()
                                ->send();
                        }
                    })
                    ->deselectRecordsAfterCompletion(),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),

            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),

                ]),
            ]);
    }

    /**
     * Detect graduation scenario based on current class level
     *
     * @param  string  $tingkat  Current class level (VII-XII)
     * @return string Scenario: 'normal', 'graduation_mts', or 'graduation_ma'
     */
    private function detectGraduationScenario(string $tingkat): string
    {
        if ($tingkat === 'IX') {
            return 'graduation_mts';
        }

        if ($tingkat === 'XII') {
            return 'graduation_ma';
        }

        return 'normal';
    }

    /**
     * Build form schema based on graduation scenario
     *
     * @param  string  $scenario  Graduation scenario
     * @param  Kelas  $currentClass  Current class object
     * @return array Form schema
     */
    private function buildFormForScenario(string $scenario, Kelas $currentClass): array
    {
        if ($scenario === 'normal') {
            return $this->buildNormalPromotionForm($currentClass);
        }

        if ($scenario === 'graduation_mts') {
            return $this->buildMtsGraduationForm($currentClass);
        }

        if ($scenario === 'graduation_ma') {
            return $this->buildMaGraduationForm($currentClass);
        }

        return [];
    }

    /**
     * Build form for normal class promotion
     */
    private function buildNormalPromotionForm(Kelas $currentClass): array
    {
        $currentLevel = $currentClass->tingkat;
        $madrasahId = $currentClass->madrasah_id;

        $levelMap = [
            'VII' => 'VIII',
            'VIII' => 'IX',
            'X' => 'XI',
            'XI' => 'XII',
        ];

        $nextLevel = $levelMap[$currentLevel] ?? null;

        if (! $nextLevel) {
            return [];
        }

        // Pre-load options untuk optimize
        // Global scope HasMadrasahScope sudah handle madrasah filtering
        $kelasOptions = Kelas::where('tingkat', $nextLevel)
            ->pluck('nama', 'id')
            ->toArray();

        return [
            Forms\Components\Select::make('new_class_id')
                ->label('Pilih Kelas Tujuan')
                ->options($kelasOptions)
                ->searchable()
                ->required()
                ->helperText("Siswa akan naik ke tingkat {$nextLevel}")
                ->placeholder('Pilih kelas tujuan'),
        ];
    }

    /**
     * Build form for MTs graduation (Class IX)
     */
    private function buildMtsGraduationForm(Kelas $currentClass): array
    {
        // Pre-load kelas X options untuk optimize dan memastikan data tersedia
        // Global scope HasMadrasahScope sudah handle madrasah filtering
        $kelasXOptions = Kelas::where('tingkat', 'X')
            ->pluck('nama', 'id')
            ->toArray();

        return [
            Forms\Components\Radio::make('graduation_type')
                ->label('Pilihan Kelulusan MTs')
                ->options([
                    'lanjut' => 'Lanjut ke MA (Kelas X)',
                    'tamat' => 'Tamat Sekolah (Tidak Melanjutkan)',
                ])
                ->required()
                ->reactive()
                ->helperText('Pilih apakah siswa akan melanjutkan ke MA atau tamat sekolah')
                ->default('lanjut'),

            Forms\Components\Select::make('new_class_id')
                ->label('Pilih Kelas X (MA)')
                ->options($kelasXOptions)
                ->searchable()
                ->required(fn ($get) => $get('graduation_type') === 'lanjut')
                ->visible(fn ($get) => $get('graduation_type') === 'lanjut')
                ->helperText('Pilih kelas X untuk siswa yang melanjutkan ke MA')
                ->placeholder('Pilih kelas tujuan'),

            Forms\Components\Textarea::make('keterangan')
                ->label('Keterangan')
                ->placeholder('Catatan tambahan (opsional)')
                ->rows(3),
        ];
    }

    /**
     * Build form for MA graduation (Class XII)
     */
    private function buildMaGraduationForm(Kelas $currentClass): array
    {
        return [
            Forms\Components\Placeholder::make('info')
                ->label('Informasi Kelulusan')
                ->content('Siswa akan ditandai sebagai LULUS MA dan tidak akan melanjutkan ke jenjang berikutnya. Data siswa akan diarsipkan sebagai alumni.'),

            Forms\Components\Textarea::make('keterangan')
                ->label('Keterangan')
                ->placeholder('Catatan tambahan (opsional)')
                ->rows(3),
        ];
    }

    /**
     * Handle normal class promotion (VII→VIII, VIII→IX, X→XI, XI→XII)
     *
     * @param  Collection  $records  Selected students
     * @param  array  $data  Form data
     * @param  TahunAjaran  $activeTahunAjaran  Active academic year
     */
    private function handleNormalPromotion(Collection $records, array $data, TahunAjaran $activeTahunAjaran): void
    {
        $newClassId = $data['new_class_id'];
        $newClass = Kelas::find($newClassId);

        DB::transaction(function () use ($records, $newClassId, $activeTahunAjaran) {
            // Update old RiwayatKelas: status = 'lulus'
            RiwayatKelas::whereIn('siswa_id', $records->pluck('id'))
                ->where('status', 'aktif')
                ->update([
                    'status' => 'lulus',
                    'keterangan' => 'Naik kelas',
                ]);

            // Update Siswa: kelas_id = new class
            Siswa::whereIn('id', $records->pluck('id'))
                ->update(['kelas_id' => $newClassId]);

            // Insert new RiwayatKelas: status = 'aktif'
            $riwayatKelasData = $records->map(function ($siswa) use ($newClassId, $activeTahunAjaran) {
                return [
                    'siswa_id' => $siswa->id,
                    'kelas_id' => $newClassId,
                    'tahun_ajaran_id' => $activeTahunAjaran->id,
                    'status' => 'aktif',
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            })->toArray();

            RiwayatKelas::insert($riwayatKelasData);
        });

        // Show success notification
        Notification::make()
            ->title('Proses Kenaikan Kelas Berhasil')
            ->body("Berhasil menaikkan {$records->count()} siswa ke kelas {$newClass->nama}.")
            ->success()
            ->send();
    }

    /**
     * Handle MTs graduation (IX → X or Tamat)
     *
     * @param  Collection  $records  Selected students
     * @param  array  $data  Form data
     * @param  TahunAjaran  $activeTahunAjaran  Active academic year
     */
    private function handleGraduationMTs(Collection $records, array $data, TahunAjaran $activeTahunAjaran): void
    {
        $graduationType = $data['graduation_type'] ?? null;
        $keterangan = $data['keterangan'] ?? '';

        if ($graduationType === 'lanjut') {
            // Branch: Lanjut ke MA
            $newClassId = $data['new_class_id'] ?? null;

            if (! $newClassId) {
                throw new \Exception('Kelas tujuan harus dipilih');
            }

            $newClass = Kelas::find($newClassId);

            if (! $newClass) {
                throw new \Exception('Kelas tidak ditemukan');
            }

            DB::transaction(function () use ($records, $newClassId, $activeTahunAjaran, $keterangan) {
                // Update old RiwayatKelas
                RiwayatKelas::whereIn('siswa_id', $records->pluck('id'))
                    ->where('status', 'aktif')
                    ->update([
                        'status' => 'lulus',
                        'keterangan' => 'Lulus MTs, melanjutkan ke MA. '.$keterangan,
                    ]);

                // Update Siswa
                Siswa::whereIn('id', $records->pluck('id'))
                    ->update(['kelas_id' => $newClassId]);

                // Insert new RiwayatKelas for MA
                $riwayatKelasData = $records->map(function ($siswa) use ($newClassId, $activeTahunAjaran) {
                    return [
                        'siswa_id' => $siswa->id,
                        'kelas_id' => $newClassId,
                        'tahun_ajaran_id' => $activeTahunAjaran->id,
                        'status' => 'aktif',
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];
                })->toArray();

                RiwayatKelas::insert($riwayatKelasData);
            });

            // Success notification for lanjut
            Notification::make()
                ->title('Proses Kelulusan MTs Berhasil')
                ->body("Berhasil meluluskan {$records->count()} siswa dari MTs dan memindahkan ke MA kelas {$newClass->nama}.")
                ->success()
                ->send();

        } else {
            // Branch: Tamat
            DB::transaction(function () use ($records, $keterangan) {
                // Update old RiwayatKelas
                RiwayatKelas::whereIn('siswa_id', $records->pluck('id'))
                    ->where('status', 'aktif')
                    ->update([
                        'status' => 'lulus',
                        'keterangan' => 'Lulus MTs, tidak melanjutkan. '.$keterangan,
                    ]);

                // NOTE: Tidak update kelas_id karena kolom tidak nullable
                // Siswa tetap tercatat di kelas terakhir mereka (kelas IX)
                // Status 'lulus' di RiwayatKelas menandakan mereka sudah tamat

                // NO new RiwayatKelas created
            });

            // Success notification for tamat
            Notification::make()
                ->title('Proses Kelulusan MTs Berhasil')
                ->body("Berhasil meluluskan {$records->count()} siswa dari MTs. Siswa tidak melanjutkan ke MA.")
                ->success()
                ->send();
        }
    }

    /**
     * Handle MA graduation (XII → Tamat)
     *
     * @param  Collection  $records  Selected students
     * @param  array  $data  Form data
     * @param  TahunAjaran  $activeTahunAjaran  Active academic year
     */
    private function handleGraduationMA(Collection $records, array $data, TahunAjaran $activeTahunAjaran): void
    {
        $keterangan = $data['keterangan'] ?? '';

        DB::transaction(function () use ($records, $keterangan) {
            // Update old RiwayatKelas
            RiwayatKelas::whereIn('siswa_id', $records->pluck('id'))
                ->where('status', 'aktif')
                ->update([
                    'status' => 'lulus',
                    'keterangan' => 'Lulus MA. '.$keterangan,
                ]);

            // NOTE: Tidak update kelas_id karena kolom tidak nullable
            // Siswa tetap tercatat di kelas terakhir mereka (kelas XII)
            // Status 'lulus' di RiwayatKelas menandakan mereka sudah tamat

            // NO new RiwayatKelas created
        });

        // Success notification
        Notification::make()
            ->title('Proses Kelulusan MA Berhasil')
            ->body("Berhasil meluluskan {$records->count()} siswa dari MA.")
            ->success()
            ->send();
    }
}
