<?php

namespace App\Filament\Resources\KamarResource\RelationManagers;

use App\Models\Kamar;
use App\Models\Siswa;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

class SiswasRelationManager extends RelationManager
{
    protected static string $relationship = 'siswas';

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('nama')
                    ->required()
                    ->maxLength(255),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->heading('Siswa')
            ->description('Kelola siswa yang tinggal di kamar ini')
            ->recordTitleAttribute('nama')
            ->modifyQueryUsing(fn ($query) => $query->aktif()) // Only show active students
            ->columns([
                Tables\Columns\TextColumn::make('nisn')
                    ->searchable()
                    ->label('NISN'),
                Tables\Columns\TextColumn::make('nama')
                    ->searchable()
                    ->label('Nama'),
                Tables\Columns\TextColumn::make('jenis_kelamin')
                    ->label('Jenis Kelamin'),
                Tables\Columns\TextColumn::make('kelas.nama')
                    ->label('Kelas')
                    ->sortable(),
            ])
            ->filters([
                //
            ])
            ->headerActions([
                // Tables\Actions\CreateAction::make(),
                Tables\Actions\BulkAction::make('pindah_kamar')
                    ->icon('heroicon-o-arrow-right-circle')
                    ->label('Pindah Kamar')
                    ->form([
                        Forms\Components\Select::make('target_room_id')
                            ->label('Pilih Kamar Tujuan')
                            ->options(function () {
                                $currentRoom = $this->getOwnerRecord();

                                $availableRooms = Kamar::query()
                                    ->where('madrasah_id', $currentRoom->madrasah_id)
                                    ->where('jenis_kelamin', $currentRoom->jenis_kelamin)
                                    ->where('id', '!=', $currentRoom->id)
                                    ->pluck('nama_kamar', 'id');

                                if ($availableRooms->isEmpty()) {
                                    Notification::make()
                                        ->title('Tidak Ada Kamar Tersedia')
                                        ->body('Tidak ada kamar lain dengan jenis kelamin yang sama di madrasah ini.')
                                        ->warning()
                                        ->send();
                                }

                                return $availableRooms;
                            })
                            ->required()
                            ->helperText('Hanya menampilkan kamar dengan jenis kelamin yang sama'),
                    ])
                    ->action(function (Collection $records, array $data) {
                        try {
                            $targetRoomId = $data['target_room_id'];
                            $targetRoom = Kamar::find($targetRoomId);

                            // Count current students in target room
                            $currentStudentsCount = Siswa::where('kamar_id', $targetRoomId)->count();

                            // Count students being transferred
                            $studentsToTransfer = $records->count();

                            // Calculate total after transfer
                            $totalAfterTransfer = $currentStudentsCount + $studentsToTransfer;

                            // Validate capacity
                            if ($totalAfterTransfer > $targetRoom->kapasitas) {
                                Notification::make()
                                    ->title('Kapasitas Tidak Cukup')
                                    ->body("Kamar tujuan memiliki {$currentStudentsCount} siswa, kapasitas {$targetRoom->kapasitas}. Anda mencoba memindahkan {$studentsToTransfer} siswa.")
                                    ->danger()
                                    ->send();

                                return;
                            }

                            // Execute transfer in transaction
                            DB::transaction(function () use ($records, $targetRoomId) {
                                // Bulk update kamar_id for selected students
                                Siswa::whereIn('id', $records->pluck('id'))
                                    ->update(['kamar_id' => $targetRoomId]);
                            });

                            // Show success notification
                            Notification::make()
                                ->title('Pemindahan Kamar Berhasil')
                                ->body("Berhasil memindahkan {$studentsToTransfer} siswa ke {$targetRoom->nama_kamar}.")
                                ->success()
                                ->send();
                        } catch (\Exception $e) {
                            // Log error for debugging
                            \Log::error('Error transferring students to room: '.$e->getMessage());

                            // Show error notification
                            Notification::make()
                                ->title('Terjadi Kesalahan')
                                ->body('Gagal memindahkan siswa. Silakan coba lagi atau hubungi administrator.')
                                ->danger()
                                ->send();
                        }
                    })
                    ->deselectRecordsAfterCompletion(),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
}
