<?php

namespace App\Filament\Pages;

use App\Models\DetailPresensiNonAsrama;
use App\Models\JenisPresensiNonAsrama;
use App\Models\Kelas;
use App\Models\PresensiNonAsrama;
use App\Models\TahunAjaran;
use BezhanSalleh\FilamentShield\Traits\HasWidgetShield;
use Filament\Actions\Action;
use Filament\Forms;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Illuminate\Support\Facades\Auth;
use Livewire\Attributes\Url;

class InputPresensiNonAsrama extends Page implements HasForms
{
    use HasWidgetShield,InteractsWithForms;

    protected static ?string $navigationIcon = 'heroicon-o-document-text';

    protected static string $view = 'filament.pages.input-presensi-non-asrama';

    protected static bool $shouldRegisterNavigation = false;

    public static function canAccess(): bool
    {
        return Auth::user()->can('page_InputPresensiNonAsrama');
    }

    #[Url]
    public ?string $kelas_id = null;

    #[Url]
    public ?string $jenis_presensi_id = null;

    #[Url]
    public ?string $tanggal = null;

    public ?array $data = [];

    public function mount(): void
    {
        $this->form->fill();

        if ($this->kelas_id && $this->jenis_presensi_id) {
            $this->loadData();
        }
    }

    public function loadData(): void
    {
        // Check if presensi already exists
        $presensi = PresensiNonAsrama::where('kelas_id', $this->kelas_id)
            ->where('jenis_presensi_non_asrama_id', $this->jenis_presensi_id)
            ->where('tanggal', $this->tanggal ?? now()->toDateString())
            ->first();

        if ($presensi) {
            // Load existing data
            $details = $presensi->detailPresensiNonAsramas;

            $this->data['students'] = $details->mapWithKeys(function ($item) {
                return [$item->siswa_id => $item->status];
            })->toArray();

            $this->data['keterangan'] = $details->mapWithKeys(function ($item) {
                return [$item->siswa_id => $item->keterangan];
            })->toArray();
        } else {
            // Initialize new data (default 'hadir')
            $kelas = Kelas::find($this->kelas_id);
            if ($kelas) {
                $this->data['students'] = $kelas->siswas->mapWithKeys(function ($item) {
                    return [$item->id => 'hadir'];
                })->toArray();
            }
        }
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('Siswa')
                    ->description('Pilih status kehadiran untuk setiap siswa. Keterangan akan muncul otomatis jika status bukan Hadir.')
                    ->schema(function () {
                        if (! $this->kelas_id) {
                            return [];
                        }

                        $kelas = Kelas::find($this->kelas_id);
                        if (! $kelas) {
                            return [];
                        }

                        $jenis = JenisPresensiNonAsrama::find($this->jenis_presensi_id);
                        // Logic for Ortu role: only show their own child
                        $user = Auth::user();
                        if ($user->hasRole(['ortu', 'Ortu'])) {
                            // Assuming User has 'siswa' relationship or we query manually
                            $mySiswa = $user->siswa; // Need to ensure User model has this relation!
                            if ($mySiswa && $mySiswa->kelas_id == $this->kelas_id) {
                                $siswas = collect([$mySiswa]);
                            } else {
                                return [Forms\Components\Placeholder::make('error')->content('Anda tidak memiliki siswa di kelas ini.')];
                            }
                        } else {
                            // Regular logic for teachers/admins
                            $siswas = $kelas->siswas;
                        }

                        // Split students into 2 columns (or 1 if single student like Ortu)
                        $total = $siswas->count();
                        if ($total === 1) {
                            $leftGroup = $siswas;
                            $rightGroup = collect([]);
                        } else {
                            $half = ceil($total / 2);
                            $leftGroup = $siswas->take($half);
                            $rightGroup = $siswas->skip($half);
                        }

                        // Left Column
                        $leftFields = [];
                        foreach ($leftGroup as $siswa) {
                            $leftFields[] = Forms\Components\Fieldset::make($siswa->nama)
                                ->schema([
                                    Forms\Components\Select::make('students.'.$siswa->id)
                                        ->label('Status Kehadiran')
                                        ->options([
                                            'hadir' => 'Hadir',
                                            'sakit' => 'Sakit',
                                            'izin' => 'Izin',
                                            'alpha' => 'Alpha',
                                        ])
                                        ->default('hadir')
                                        ->required()
                                        ->live()
                                        ->native(false)
                                        ->columnSpanFull(),
                                    Forms\Components\Textarea::make('keterangan.'.$siswa->id)
                                        ->label('Keterangan')
                                        ->placeholder('Masukkan keterangan (opsional)')
                                        ->rows(2)
                                        ->columnSpanFull()
                                        ->visible(fn (Forms\Get $get) => $get('students.'.$siswa->id) !== 'hadir'
                                        ),
                                ])
                                ->columns(1);
                        }

                        // Right Column
                        $rightFields = [];
                        foreach ($rightGroup as $siswa) {
                            $rightFields[] = Forms\Components\Fieldset::make($siswa->nama)
                                ->schema([
                                    Forms\Components\Select::make('students.'.$siswa->id)
                                        ->label('Status Kehadiran')
                                        ->options([
                                            'hadir' => 'Hadir',
                                            'sakit' => 'Sakit',
                                            'izin' => 'Izin',
                                            'alpha' => 'Alpha',
                                        ])
                                        ->default('hadir')
                                        ->required()
                                        ->live()
                                        ->native(false)
                                        ->columnSpanFull(),
                                    Forms\Components\Textarea::make('keterangan.'.$siswa->id)
                                        ->label('Keterangan')
                                        ->placeholder('Masukkan keterangan (opsional)')
                                        ->rows(2)
                                        ->columnSpanFull()
                                        ->visible(fn (Forms\Get $get) => $get('students.'.$siswa->id) !== 'hadir'
                                        ),
                                ])
                                ->columns(1);
                        }

                        return [
                            Forms\Components\Grid::make(2)
                                ->schema([
                                    Forms\Components\Section::make()
                                        ->schema($leftFields)
                                        
                                        ->visible(count($leftFields) > 0),
                                  
                                ]),
                        ];
                    })
                    ->collapsible(),
            ])
            ->statePath('data');
    }

    public function save(): void
    {
        $tahunAjaran = TahunAjaran::where('is_aktif', true)->first();

        if (! $tahunAjaran) {
            Notification::make()->title('Tahun Ajaran Aktif tidak ditemukan')->danger()->send();

            return;
        }

        // Find or Create Presensi Header
        $presensi = PresensiNonAsrama::firstOrCreate(
            [
                'kelas_id' => $this->kelas_id,
                'jenis_presensi_non_asrama_id' => $this->jenis_presensi_id,
                'tanggal' => $this->tanggal ?? now()->toDateString(),
            ],
            [
                'tahun_ajaran_id' => $tahunAjaran->id,
                'user_id' => Auth::id(),
                'is_locked' => false,
            ]
        );

        // Save Details
        // Ensure we only save permitted students for Ortu
        $user = Auth::user();
        $allowedSiswaIds = $this->data['students'];

        // Security check for Ortu: Don't allow saving other students via hidden field hacks
        if ($user->hasRole(['ortu', 'Ortu'])) {
            $mySiswa = $user->siswa;
            if (! $mySiswa) {
                return;
            }
            // Only process the specific student ID
            $allowedSiswaIds = array_intersect_key($this->data['students'], [$mySiswa->id => '']);
        }

        foreach ($allowedSiswaIds as $siswaId => $status) {
            $keterangan = $this->data['keterangan'][$siswaId] ?? null;

            DetailPresensiNonAsrama::updateOrCreate(
                [
                    'presensi_non_asrama_id' => $presensi->id,
                    'siswa_id' => $siswaId,
                ],
                [
                    'status' => $status,
                    'keterangan' => $keterangan,
                ]
            );
        }

        Notification::make()
            ->title('Presensi berhasil disimpan')
            ->success()
            ->send();

        // Redirect back or to dashboard
        $this->redirectRoute('filament.admin.pages.dashboard');
    }

    public function getHeaderActions(): array
    {
        return [
            Action::make('save')
                ->label('Simpan Presensi')
                ->action('save')
                ->color('primary'),
        ];
    }

    public function getTitle(): string
    {
        if ($this->kelas_id && $this->jenis_presensi_id) {
            $kelas = Kelas::find($this->kelas_id);
            $jenis = JenisPresensiNonAsrama::find($this->jenis_presensi_id);

            if ($kelas && $jenis) {
                return "Input Presensi: {$jenis->nama} - {$kelas->nama}";
            }
        }

        return 'Input Presensi Non Asrama';
    }
}
