<?php

namespace App\Filament\Pages;

use Filament\Forms;
use App\Models\Kamar;
use Filament\Forms\Form;
use Filament\Pages\Page;
use App\Models\TahunAjaran;
use Illuminate\Support\Str;
use Filament\Actions\Action;
use Livewire\Attributes\Url;
use App\Models\PresensiAsrama;
use App\Models\JenisPresensiAsrama;
use App\Models\DetailPresensiAsrama;
use Illuminate\Support\Facades\Auth;
use Filament\Forms\Contracts\HasForms;
use Filament\Notifications\Notification;
use Filament\Forms\Concerns\InteractsWithForms;
use BezhanSalleh\FilamentShield\Traits\HasWidgetShield;

class InputPresensiAsrama extends Page implements HasForms
{
    use HasWidgetShield,InteractsWithForms;

    protected static ?string $navigationIcon = 'heroicon-o-document-text';

    protected static string $view = 'filament.pages.input-presensi-asrama';

    protected static bool $shouldRegisterNavigation = false;

    public static function canAccess(): bool
    {
        return Auth::user()->can('page_InputPresensiAsrama');
    }

    #[Url]
    public ?string $kamar_id = null;

    #[Url]
    public ?string $jenis_presensi_id = null;

    #[Url]
    public ?string $tanggal = null;

    public ?array $data = [];

    public function mount(): void
    {
        $this->form->fill();

        if ($this->kamar_id && $this->jenis_presensi_id) {
            $this->loadData();
        }
    }

    public function loadData(): void
    {
        // Check if presensi already exists
        $presensi = PresensiAsrama::where('kamar_id', $this->kamar_id)
            ->where('jenis_presensi_asrama_id', $this->jenis_presensi_id)
            ->where('tanggal', $this->tanggal ?? now()->toDateString())
            ->first();

        if ($presensi) {
            // Load existing data
            $details = $presensi->detailPresensiAsramas;

            $this->data['students'] = $details->mapWithKeys(function ($item) {
                return [$item->siswa_id => $item->status];
            })->toArray();

            $this->data['keterangan'] = $details->mapWithKeys(function ($item) {
                return [$item->siswa_id => $item->keterangan];
            })->toArray();
        } else {
            // Initialize new data (default 'hadir')
            $kamar = Kamar::find($this->kamar_id);
            if ($kamar) {
                $this->data['students'] = $kamar->siswas->mapWithKeys(function ($item) {
                    return [$item->id => 'hadir'];
                })->toArray();
            }
        }
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('Daftar Santri')
                    ->description('Pilih status kehadiran untuk setiap santri. Keterangan akan muncul otomatis jika status bukan Hadir/Berpuasa.')
                    ->schema(function () {
                        if (! $this->kamar_id) {
                            return [];
                        }

                        $kamar = Kamar::find($this->kamar_id);
                        if (! $kamar) {
                            return [];
                        }

                        $jenis = JenisPresensiAsrama::find($this->jenis_presensi_id);
                        $isPuasa = $jenis && Str::contains($jenis->nama, 'Puasa');

                        $siswas = $kamar->siswas;

                        // Split students into 2 groups
                        $total = $siswas->count();
                        $half = ceil($total / 2);
                        $leftGroup = $siswas->take($half);
                        $rightGroup = $siswas->skip($half);

                        // Left Column
                        $leftFields = [];
                        foreach ($leftGroup as $siswa) {
                            $leftFields[] = Forms\Components\Fieldset::make($siswa->nama)
                                ->schema([
                                    Forms\Components\Select::make('students.'.$siswa->id)
                                        ->label('Status Kehadiran')
                                        ->options([
                                            'hadir' => $isPuasa ? 'Berpuasa' : 'Hadir',
                                            'sakit' => 'Sakit',
                                            'izin' => $isPuasa ? 'Berhalangan (Haid)' : 'Izin',
                                            'alpha' => $isPuasa ? 'Tidak Berpuasa' : 'Alpha',
                                        ])
                                        ->default('hadir')
                                        ->required()
                                        ->live()
                                        ->native(false)
                                        ->columnSpanFull(),
                                    Forms\Components\Textarea::make('keterangan.'.$siswa->id)
                                        ->label('Keterangan')
                                        ->placeholder('Masukkan keterangan (opsional)')
                                        ->rows(2)
                                        ->columnSpanFull()
                                        ->visible(fn (Forms\Get $get) => $get('students.'.$siswa->id) !== 'hadir'
                                        ),
                                ])
                                ->columns(1);
                        }

                        // Right Column
                        $rightFields = [];
                        foreach ($rightGroup as $siswa) {
                            $rightFields[] = Forms\Components\Fieldset::make($siswa->nama)
                                ->schema([
                                    Forms\Components\Select::make('students.'.$siswa->id)
                                        ->label('Status Kehadiran')
                                        ->options([
                                            'hadir' => $isPuasa ? 'Berpuasa' : 'Hadir',
                                            'sakit' => 'Sakit',
                                            'izin' => $isPuasa ? 'Berhalangan (Haid)' : 'Izin',
                                            'alpha' => $isPuasa ? 'Tidak Berpuasa' : 'Alpha',
                                        ])
                                        ->default('hadir')
                                        ->required()
                                        ->live()
                                        ->native(false)
                                        ->columnSpanFull(),
                                    Forms\Components\Textarea::make('keterangan.'.$siswa->id)
                                        ->label('Keterangan')
                                        ->placeholder('Masukkan keterangan (opsional)')
                                        ->rows(2)
                                        ->columnSpanFull()
                                        ->visible(fn (Forms\Get $get) => $get('students.'.$siswa->id) !== 'hadir'
                                        ),
                                ])
                                ->columns(1);
                        }

                        return [
                            Forms\Components\Grid::make(2)
                                ->schema([
                                    Forms\Components\Section::make('Kolom Kiri')
                                        ->schema($leftFields)
                                        ->columnSpan(1),
                                    Forms\Components\Section::make('Kolom Kanan')
                                        ->schema($rightFields)
                                        ->columnSpan(1),
                                ]),
                        ];
                    })
                    ->collapsible(),
            ])
            ->statePath('data');
    }

    public function save(): void
    {
        $tahunAjaran = TahunAjaran::where('is_aktif', true)->first();

        if (! $tahunAjaran) {
            Notification::make()->title('Tahun Ajaran Aktif tidak ditemukan')->danger()->send();

            return;
        }

        // Find or Create Presensi Header
        $presensi = PresensiAsrama::firstOrCreate(
            [
                'kamar_id' => $this->kamar_id,
                'jenis_presensi_asrama_id' => $this->jenis_presensi_id,
                'tanggal' => $this->tanggal ?? now()->toDateString(),
            ],
            [
                'tahun_ajaran_id' => $tahunAjaran->id,
                'user_id' => Auth::id(),
                'is_locked' => false,
            ]
        );

        // Save Details
        foreach ($this->data['students'] as $siswaId => $status) {
            $keterangan = $this->data['keterangan'][$siswaId] ?? null;

            DetailPresensiAsrama::updateOrCreate(
                [
                    'presensi_asrama_id' => $presensi->id,
                    'siswa_id' => $siswaId,
                ],
                [
                    'status' => $status,
                    'keterangan' => $keterangan,
                ]
            );
        }

        Notification::make()
            ->title('Presensi berhasil disimpan')
            ->success()
            ->send();

        $this->redirectRoute('filament.admin.pages.dashboard');
    }

    public function getHeaderActions(): array
    {
        return [
            Action::make('save')
                ->label('Simpan Presensi')
                ->action('save')
                ->color('primary'),
        ];
    }

    public function getTitle(): string
    {
        if ($this->kamar_id && $this->jenis_presensi_id) {
            $kamar = Kamar::find($this->kamar_id);
            $jenis = JenisPresensiAsrama::find($this->jenis_presensi_id);

            return "Input Presensi: {$jenis->nama} - {$kamar->nama_kamar}";
        }

        return 'Input Presensi Asrama';
    }
}
