<?php

namespace App\Exports;

use Illuminate\Contracts\View\View;
use Maatwebsite\Excel\Concerns\FromView;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithDrawings;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use PhpOffice\PhpSpreadsheet\Worksheet\Drawing;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class SiswaExcelExport implements FromView, ShouldAutoSize, WithDrawings, WithEvents, WithStyles
{
    protected $records;

    protected $madrasah;

    public function __construct($records)
    {
        $this->records = $records;
        // Ambil madrasah dari record pertama atau dari auth user
        $this->madrasah = $records->first()?->madrasah ?? auth()->user()->madrasah;
    }

    public function view(): View
    {
        return view('exports.siswa', [
            'records' => $this->records,
            'madrasah' => $this->madrasah,
            'tanggal' => now()->locale('id')->translatedFormat('d F Y'),
        ]);
    }

    public function drawings()
    {
        // Cek apakah ada logo madrasah
        $logoPath = public_path('images/logo.png');

        if (! file_exists($logoPath)) {
            return [];
        }

        $drawing = new Drawing;
        $drawing->setName('Logo');
        $drawing->setDescription('Logo Madrasah');
        $drawing->setPath($logoPath);
        $drawing->setHeight(60); // Tinggi logo dalam pixel
        $drawing->setCoordinates('A1'); // Posisi logo
        $drawing->setOffsetX(10);
        $drawing->setOffsetY(5);

        return [$drawing];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            // Style untuk header kop surat
            1 => ['font' => ['bold' => true, 'size' => 14]],
            2 => ['font' => ['size' => 12]],

            // Style untuk header tabel
            6 => [
                'font' => ['bold' => true],
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => ['rgb' => 'E2E8F0'],
                ],
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                    ],
                ],
            ],
        ];
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function (AfterSheet $event) {
                $sheet = $event->sheet->getDelegate();
                $highestRow = $sheet->getHighestRow();

                // Set tinggi baris pertama untuk logo
                $sheet->getRowDimension(1)->setRowHeight(50);

                // Format kolom angka sebagai text
                // C = NISN, D = NIS, E = NIK, M = No KK, O = NIK Ayah, S = NIK Ibu, V = No HP Ortu
                $textColumns = ['C', 'D', 'E', 'M', 'O', 'S', 'V'];

                foreach ($textColumns as $col) {
                    $sheet->getStyle($col.'7:'.$col.$highestRow)
                        ->getNumberFormat()
                        ->setFormatCode(NumberFormat::FORMAT_TEXT);

                    // Re-set nilai sebagai string untuk memaksa format text
                    for ($row = 7; $row <= $highestRow; $row++) {
                        $value = $sheet->getCell($col.$row)->getValue();
                        if ($value) {
                            $sheet->setCellValueExplicit(
                                $col.$row,
                                $value,
                                \PhpOffice\PhpSpreadsheet\Cell\DataType::TYPE_STRING
                            );
                        }
                    }
                }
            },
        ];
    }
}
